#ifndef ARCANE__LEVEL_H
#define ARCANE__LEVEL_H

/*
 * arcane - A rogue-like game engine
 * Copyright (C) 2005  Petr Baudis <pasky@ucw.cz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <vector>
#include <list>

#include "exception.h"
#include "term.h"
#include "util.h"


class Creature;
class MapObject;

class Tile {
public:
	Tile();

	enum Type {
		BLACK = ' ',
		ROCK = '#',
		FLOOR = '.',
	};
	PROP_SIMPLE(enum Type, type);
	PROP_SIMPLE(bool, known);
	PROP_SIMPLE(bool, illum);

	bool opaque() const;

	void enter(MapObject *obj);
	void leave(MapObject *obj);
	Creature *occupied() const; // living being there
	std::list<MapObject *> objs;

	void draw(Term &term, int base_x, int base_y) const;
};


class Creature;
class ItemOnMap;
class Monster;
class Game;

class Level {
public:
	Level(int depth, int width = 80, int height = 22, int maxmonsters = 20, int maxitems = 6, int danger = -1);

	Tile operator()(unsigned x, unsigned y) const;
	Tile &operator()(unsigned x, unsigned y);

	EXCEPTION(X_baddir, (enum Direction dir), "Bad direction: " + tostr(long(dir)));

	void in_dir(unsigned &x, unsigned &y, enum Direction dir) const;
	void randpos(unsigned &x, unsigned &y) const;

	void draw(Term &term, int base_x, int base_y) const;
	void pretick(Game &game);
	void posttick(Game &game);

	// visible monsters
	std::list<Creature *> explore(unsigned x, unsigned y, int range);
	void unleash(unsigned x, unsigned y);

	PROP_SIMPLE_RO(int, depth);
	int width() const { return map_.size(); }
	int height() const { return map_[0].size(); }
	PROP_SIMPLE(int, danger);

	PROP_SIMPLE(unsigned, maxmonsters);
	EXCEPTION(X_monsterpack, (), "Multiple monsters spawned");
	Monster *spawn_monster(); // can throw X_monsterpack
	void spawn_monsters();

	PROP_SIMPLE_RO(unsigned, maxitems);
	EXCEPT_BASE1(X_itemheap, (int items_created), "Multiple items created"),
		items_created(items_created) { }
		public: int items_created;
	EXCEPT_BASE2
	ItemOnMap *scatter_item(); // can throw X_itemheap
	void scatter_items();

	void enter(Creature *creature);
	void leave(Creature *creature);
	std::list<Creature *> life;

	void add_illum(MapObject *illum);
	void del_illum(MapObject *illum);
	std::list<MapObject *> illum;

private:
	std::vector<std::vector<Tile> > map_; // [x][y]
	std::vector<std::vector<Tile> > map_view_; // how the player sees it

	unsigned long recalc_creat_prob(), recalc_item_prob();
	unsigned long total_creat_prob_, total_item_prob_;

	// Fails if there is an obstacle in the way.
	bool trace_visibility(unsigned x1, unsigned y1, unsigned x2, unsigned y2, void (Level::*hook)(unsigned, unsigned, void *), void *d);
	void percept(unsigned x, unsigned y, /*std::list<Creature *>*/void *creatures);
	void illum_hook(unsigned x, unsigned y, void *);
	void recalc_illum();
};



inline
Tile::Tile()
	: type_(ROCK), known_(false), illum_(false)
{
}

inline bool
Tile::opaque() const
	{ return type_ != Tile::ROCK; }


inline Tile &
Level::operator ()(unsigned x, unsigned y)
{
	if (x >= 80 || y >= 22) throw;
	return map_[x][y];
}
inline Tile
Level::operator ()(unsigned x, unsigned y) const
	{ return (*this)(x, y); }

inline void
Level::unleash(unsigned x, unsigned y)
	{ map_view_[x][y] = map_[x][y]; map_view_[x][y].known(true); }

inline void
Level::add_illum(MapObject *obj)
	{ illum.push_back(obj); }
inline void
Level::del_illum(MapObject *obj)
	{ illum.remove(obj); }

#endif
