#ifndef ARCANE__CREATURE_H
#define ARCANE__CREATURE_H

/*
 * arcane - A rogue-like game engine
 * Copyright (C) 2005  Petr Baudis <pasky@ucw.cz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stack>

#include "cinfo.h"
#include "level.h"
#include "mapobj.h"


class Item;
class Game;

typedef unsigned long xp_T;

class Creature: public MapObject {
public:
	Creature(CreatureInfo &cinfo, Level *level, unsigned x, unsigned y);
	virtual ~Creature();

	EXCEPT_BASE1(X_creatdir, (enum Direction dir, Creature *c), "Creature in direction: " + tostr(long(dir))),
		c(c) { }
		public: Creature *c;
	EXCEPT_BASE2

	virtual void tick(Game &game);

	virtual void move(enum Direction dir);
	virtual void die(std::string &msg);
	virtual bool hit(int points, std::string &msg);

	virtual void pick();
	virtual void drop();

	void backpack_push(Item *item);
	Item *backpack_pop(); /* a non-put pointer */

	virtual bool player() const = 0;
	const std::string &name() const;
	virtual enum Color color() const;

	bool karmic() const;

	unsigned visible_range() const;

	PROP_WHOOKED(int, st, recalc_stats());
	PROP_WHOOKED(int, dx, recalc_stats());
	PROP_WHOOKED(int, to, recalc_stats());
	PROP_WHOOKED(int, pe, recalc_stats());

	int dv() const;
	int pv() const;
	int av() const;
	PROP_SIMPLE_RO(int, hp);
	PROP_SIMPLE_RO(int, maxhp);
	PROP_SIMPLE_RO(int, speed);

	PROP_SIMPLE_RO(int, height);
	PROP_SIMPLE_RO(int, weight);

	PROP_SIMPLE_RO(xp_T, xp);
	long logxp() const;
	void xp_gain(xp_T xp);
	// successful...
	xp_T xp_worth_attack();
	xp_T xp_worth_defend();

private:
	CreatureInfo &cinfo_;
	void recalc_stats();

	std::stack<Item *> backpack_;

	int healrate_;
	int healticker_;

	int xpticker_;
};

#define FF_NORMAL	0
#define FF_BYPASS_DV	1
#define FF_BYPASS_PV	2
extern Creature *fight(Creature &a, Creature &d, std::string &msg, unsigned long flags);

inline const std::string &
Creature::name() const
	{ return cinfo_.name; }

inline bool
Creature::karmic() const
	{ return (cinfo_.flags & CF_KARMIC); }

inline unsigned
Creature::visible_range() const
{
	return pe(); // / 2;
}

inline enum Color
Creature::color() const
{
	if (karmic())
		return enum Color(8 + rand(8));
	return MapObject::color();
}

#endif
