/* Socket tools */

#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <unistd.h>

#include "addr.h"
#include "bufio.h"
#include "select.h"
#include "socket.h"


/*
 * All functions creating sockets and manipulating explicitly with
 * sock_t structures return sock_t * on success, NULL on failure.
 *
 * All functions manipulating with sockets return 1 on success
 * and 0 on failure.
 *
 * Unless explicitly noted otherwise, obviously.
 */

sock_t *
sock_del(sock_t *sock) {
	if (! sock)
		return NULL;

	if (sock->addr)
		sock_addr_del(sock->addr);

	if (sock->buf)
		bufio_del(sock->buf);

	if (sock->fd >= 0)
		close(sock->fd);

	free(sock);

	return NULL;
}

sock_t *
sock_get(int fd) {
	sock_t *sock = malloc(sizeof(sock_t));

	if (! sock)
		return NULL;

	sock->fd = fd;
	sock->state = SS_VOID;
	sock->flags = SF_NONE;
	sock->addr = NULL;

	sock->handler_r = sock->handler_w = sock->handler_e = NULL;
	sock->handler_r_data = sock->handler_w_data = sock->handler_e_data = NULL;

	sock->buf = NULL;

	sock->set = NULL;

	sock->next = NULL;

	return sock;
}

sock_t *
sock_get4() {
	int fd = socket(PF_INET, SOCK_STREAM, 0);

	if (fd < 0)
		return NULL;

	return sock_get(fd);
}

int
sock_set_nb(sock_t *sock) {
	if (fcntl(sock->fd, F_SETFL, O_NONBLOCK) < 0)
		return 0;

	sock->flags |= SF_NBLK;

	return 1;
}

int
sock_unset_nb(sock_t *sock) {
	int fl = fcntl(sock->fd, F_GETFL);

	if ((fl < 0) || (fcntl(sock->fd, F_SETFL, fl&~O_NONBLOCK) < 0))
		return 0;

	sock->flags &= ~SF_NBLK;

	return 1;
}

int
sock_bind(sock_t *sock, addr_t *addr) {
	if (bind(sock->fd, addr->addr, addr->addrlen) < 0)
		return 0;

	sock->flags |= SF_BOUND;

	return 1;
}

sock_t *
sock_listen_addr(addr_t *addr, int nonblock) {
	sock_t *sock = sock_get4();

	if (! sock)
		return sock_addr_del(addr), NULL;

	sock->addr = addr;

	if (nonblock && ! sock_set_nb(sock))
		return sock_del(sock);

	if (! sock_bind(sock, addr))
		return sock_del(sock);

	if (listen(sock->fd, 0) < 0)
		return sock_del(sock);

	sock->state = SS_LISTEN;

	return sock;
}

sock_t *
sock_listen(char *hostname, int port, int nonblock) {
	addr_t *addr = sock_addr_getbyname(hostname, port);

	if (! addr)
		return NULL;

	return sock_listen_addr(addr, nonblock);
}

sock_t *
sock_connect_addr(addr_t *addr, int nonblock) {
	sock_t *sock = sock_get4();

	if (! sock)
		return sock_addr_del(addr), NULL;

	sock->addr = addr;

	if (nonblock && ! sock_set_nb(sock))
		return sock_del(sock);

	if (connect(sock->fd, addr->addr, addr->addrlen) < 0) {
		if (! nonblock || errno != EINPROGRESS)
			return sock_del(sock);
		else
			sock->state = SS_CONNECT;
	} else
		sock->state = SS_IDLE;

	return sock;
}

sock_t *
sock_connect(char *hostname, int port, int nonblock) {
	addr_t *addr = sock_addr_getbyname(hostname, port);

	if (! addr)
		return NULL;

	return sock_connect_addr(addr, nonblock);
}

sock_t *
sock_accept(sock_t *sock) {
	sock_t *newsock;
	int fd;
	struct sockaddr_in saddr;
	unsigned int addrlen = sizeof(saddr);

	if (sock->state != SS_LISTEN)
		return NULL;

	fd = accept(sock->fd, (struct sockaddr *)&saddr, &addrlen);

	if (fd < 0)
		return NULL;

	newsock = sock_get(fd);

	if (! newsock)
		return NULL;

	newsock->state = SS_IDLE;
	newsock->flags = sock->flags;
	newsock->handler_r = sock->handler_r;
	newsock->handler_w = sock->handler_w;
	newsock->handler_e = sock->handler_e;
	newsock->handler_r_data = sock->handler_r_data;
	newsock->handler_w_data = sock->handler_w_data;
	newsock->handler_e_data = sock->handler_e_data;
	newsock->addr = sock_addr_get(AF_INET, (struct sockaddr *)&saddr, addrlen);

	return newsock;
}

int
sock_sethandler_r(sock_t *sock, void (*handler)(sock_t *, void *), void *data) {
	sock->handler_r = handler;
	sock->handler_r_data = data;

	if (sock->set)
		sockset_notify(sock->set, sock);

	return 1;
}

int
sock_sethandler_w(sock_t *sock, void (*handler)(sock_t *, void *), void *data) {
	sock->handler_w = handler;
	sock->handler_w_data = data;

	if (sock->set)
		sockset_notify(sock->set, sock);

	return 1;
}

int
sock_sethandler_e(sock_t *sock, void (*handler)(sock_t *, void *), void *data) {
	sock->handler_e = handler;
	sock->handler_e_data = data;

	if (sock->set)
		sockset_notify(sock->set, sock);

	return 1;
}
