/** sock - the sockets abstraction */

#ifndef LIBAIO__SOCKET_H
#define LIBAIO__SOCKET_H

#include <stdint.h>

/*!#*/ typedef struct SOCK_T sock_t; /* the includes below rely on this */

#include "addr.h"
#include "bufio.h"
#include "select.h"


/*** SYNOPSIS */
/*| sock_t *s = sock_connect("example.com", 1234, 0); */


/** The socket structure */

/* Obviously this needn't be a socket. It can be a pipe or even an ordinary
 * file (not that there would be any point in doing that). Just build it up
 * using sock_get() and keep addr NULL and states && flags sane. Maybe I will
 * add a flag for it later. */
/*!typedef struct {*/ struct SOCK_T {
	int fd;			/* MUST be valid */
	unsigned int state;	/* Only one at a time; see below */
	unsigned int flags;	/* More of 'em ORed; see below */

	addr_t *addr;		/* Listening or peer address, respectively */

	/* Handlers of various socket events; any may be NULL */
	void (*handler_r)(struct SOCK_T *sock, void *data);
	void (*handler_w)(struct SOCK_T *sock, void *data);
	void (*handler_e)(struct SOCK_T *sock, void *data);

	void *handler_r_data;
	void *handler_w_data;
	void *handler_e_data;

	buf_t *buf;		/* Bufio buffer, NULL if bufio is not used */

	/* Sockset owning this socket; may be NULL if not part of a sockset */
	sockset_t *set;
	struct SOCK_T *next;
/*!} sock_t;*/ };

/** Socket states */
#define	SS_VOID		0
#define	SS_CONNECT	1
#define	SS_LISTEN	2
#define	SS_IDLE		3
#define	SS_READABLE	4
#define	SS_WRITABLE	5
#define	SS_EXCEPTION	6

/** Socket flags */
#define	SF_NONE		0
#define	SF_NBLK		1	/* Non-blocking */
#define	SF_BOUND	2	/* Bound to an addr */
#define	SF_BUFFER	4	/* Buffered i/o acting upon */


/*+ Socket functions */

/* Get an IPv4 socket. */
/* Returns a newly allocated sock_t structure with fd=socket() or NULL */
sock_t *sock_get4();

/* Get a socket structure pointing to int (fd). */
/* Returns a newly allocated sock_t structure or NULL */
sock_t *sock_get(int);

/* Delete a socket */
/* Always returns NULL */
sock_t *sock_del(sock_t *);

/* Set the socket as non-blocking */
/* Returns 1 on success, 0 otherwise */
int sock_set_nb(sock_t *);

/* Set the socket as blocking */
/* Returns 1 on success, 0 otherwise */
int sock_unset_nb(sock_t *);

/* Bind the socket to a specified address */
/* Returns 1 on success, 0 otherwise */
int sock_bind(sock_t *sock, addr_t *addr);

/* Create a socket connect to a specified host:port, optionally in
 * a non-blocking manner. */
/* Returns sock_t structure pointing to that socket, or NULL on failure */
sock_t *sock_connect(char *addr, int port, int nb);
sock_t *sock_connect_addr(addr_t *addr, int nb);

/* Open a socket listening on a specified host:port, optionally in
 * a non-blocking manner. */
/* Returns sock_t structure pointing to that socket, or NULL on failure */
sock_t *sock_listen(char *addr, int port, int nb);
sock_t *sock_listen_addr(addr_t *addr, int nb);

/* Try to accept new connection on the socket. */
/* Returns sock_t structure pointing to new socket (inheriting flags and
 * handlers from his ancestor), or NULL on failure (or when no new connection
 * is available and the socket is non-blocking). */
sock_t *sock_accept(sock_t *sock);

/* Set handlers for the socket events. These handlers are called when the
 * event is triggered on the socket and noticed by sockset_check().
 * These functions just assign the handler and call sockset_notify() -
 * obviously you can do that on your own, but more functionality can be added
 * here in the future. */
/* Returns 1 if ok, 0 currently never ;-) */
int sock_sethandler_r(sock_t *sock, void (*)(sock_t *sock, void *data), void *data);
int sock_sethandler_w(sock_t *sock, void (*)(sock_t *sock, void *data), void *data);
int sock_sethandler_e(sock_t *sock, void (*)(sock_t *sock, void *data), void *data);

#endif
