#ifndef _LANGUAGE_H
#define _LANGUAGE_H
//
// This file is part of TTDPatch
// Copyright (C) 1999 by Josef Drexler
//
// language.h: defines variables for multi-lingual support
//

//
// To add a string:
// - scroll down and find LANG_LASTSTRING
// - write down that number
// - increase LANG_LASTSTRING by one
// - define your constant to have the previous value of
//   LANG_LASTSTRING, the one you've written down
//

#include <stdlib.h>

#include "types.h"

#ifndef C
#	define C	// for common.h
#endif

#include "common.h"

#if defined(IS_LANGUAGE_CPP)
#	define ISEXTERN extern
#elif defined(IS_MAKELANG_CPP)
#	define ISEXTERN
#elif defined(IS_TTDSTART_CPP)
#	define ISEXTERN
#elif defined(IS_LOADLNG_H)
#	define ISEXTERN extern
#elif defined(IS_SWITCHES_CPP)
#	define ISEXTERN extern
#elif defined(IS_CHECKEXE_CPP)
#	define ISEXTERN extern
#else
#	error Something is wrong
#endif

// Magic code for the language data. In the actual file it'll be XOR'd
// with 32 (i.e., in lower case) so that the in-memory location isn't
// found while searching the exe file for the language data
#define LANGCODELEN 8		// number of bytes in magic code
#define LANGINFOSIZE 16		// size of each entry in the language info
#define LANGMAXSIZEOFS (LANGCODELEN+2*4)	// offset to max.buffer sizes
#define LANGINFOOFFSET (LANGMAXSIZEOFS+2*4)	// offset of language info

static char langcode[LANGCODELEN+1] = "LangData";

#define LANG_MAX 10	// total number of languages

// Structure of language.dat
// 	Offset	Length	Content
// 	0	8(a)	langcode[]	(a) LANGCODELEN really
//	8(a)	4	TTDPatch Version this language data is for
// 	12	4	Number of languages=#
//	16	4	Largest uncompressed buffer
//	20	4	Largest compressed buffer
//	24(b)	#*12(c)	Language info	(b) Better use LANGINFOOFFSET
//					(c) LANGINFOSIZE really
//
// Language info (starting at LANGINFOOFFSET)
//	RelOfs	Length	Content
//	0	4	Relative offset to language data, counted from
//			offset of langcode[]
//	4	4	Relative offset to country codes
//	8	4	Compressed size of language data (w/o country codes)
//	12	4	Uncompressed size of language data (w/o country codes)
//
// Language country code info (uncompressed):
//	RelOfs	Length	Content
//	0	4	Number of country codes for which to use this language
//	4	4*#	Country codes
//
// Language data: each string has the following format
//	RelOfs	Length	Content
//	0	1	Info*
//      ?	1	Extra length**
//	?	2	New code if not consecutive
//	?	*	String data without trailing zero
//
//	Info:	This is bit-encoded to save some space
//		Bits 0-5: lower 6 bits of the string length
//		Bit 6:	  if set, the code is not consecutive, and a new
//			  code is given (if present, after the extra length)
//		Bit 7:	  the length doesn't fit in 6 bits, the next eight
//			  bits are in the byte immediately following this one
//	Extra length:	  If Info has bit 7 set, this byte contains bits
//			  6 to 13 of the string length
//	New code:	  If the codes are not consecutive, i.e. if this code
//			  is not one higher than the previous one, then this
//			  contains the new code.
//
//	Note that the code sequence is assumed to start with LANGCODE_NAME(0)
//	containing the language name.  If this is the case, the first string
//	will not have bit 6 set.
//
// Arrays are prefixed with a two-byte number telling the number of entries
//


// Insert ONLY language texts or comments between the BEGIN and END lines
// BEGIN LANGUAGE TEXTS

//-------------------------------------------
//  PROGRAM BLURBS
//-------------------------------------------

// First line of output is something like "TTDPatch V1.5.1 starting.\n"
// The program name and version are autogenerated, only put the " starting\n"
// here
#define LANG_STARTING 0


//-------------------------------------------
//  VERSION CHECKING
//-------------------------------------------

// In the version identifier, this is for the file size
#define LANG_SIZE 1

// Shown if the version is recognized
#define LANG_KNOWNVERSION 2

// Warning if the version isn't recognized.  Be sure *not* to use tabs
// in the text.  All but the last lines should end in *\n" \*
#define LANG_WRONGVERSION 3

// Keys which continue loading after the above warning. *MUST* be lower case.
#define LANG_YESKEYS 4

// Answering anything but the above keys gives this message.
#define LANG_ABORTLOAD 5

// otherwise continue loading
#define LANG_CONTINUELOAD 6

// Warning if '-y' was used and the version is unknown
#define LANG_WARNVERSION 7

// Successful load of ttdload and writing the .ver file gives this
#define LANG_VERSIONINFOSAVED 8

// Something went wrong, not all offsets could be determined
#define LANG_VERSIONINFONOTSAVED 9

// Text just after the above messages
// %s is the file name
#define LANG_WRITINGINFO 10


// -------------------------------------------
//    CREATING AND PATCHING TTDLOAD
// -------------------------------------------

// TTDLOAD.EXE doesn't exist
#define LANG_FILENOTFOUND 11

// neither do tycoon.exe or ttdx.exe.  %s is TTDX.EXE
#define LANG_NOFILESFOUND 12

// Shown when copying tycoon.exe or ttdx.exe (first %s) to ttdload.exe (2nd %s)
#define LANG_SHOWCOPYING 13

// Error if running the copy command fails.  %s is the command.
#define LANG_COPYERROR_RUN 14

// Error if command returned successfully, but nothing was copied.
// %s=TTDLOAD.EXE
#define LANG_COPYERROR_NOEXIST 15

// Invalid .EXE format
#define LANG_INVALIDEXE 16

// Version could not be determined
#define LANG_VERSIONUNCONFIRMED 17

// Shows program name (1st %s) and version (2nd %s)
#define LANG_PROGANDVER 18

// More than three numbers in the version string (not #.#.#)
#define LANG_TOOMANYNUMBERS 19

// .EXE is not TTD
#define LANG_WRONGPROGRAM 20

// Displays the parsed version number
#define LANG_PARSEDVERSION 21

// The exe has been determined to be the DOS extended executable
#define LANG_ISDOSEXTEXE 22

// The exe has been determined to be the Windows executable
#define LANG_ISWINDOWSEXE 23

// The exe is neither DOS extended nor Windows?
#define LANG_ISUNKNOWNEXE 90

// The exe is the wrong type i.e. DOS/Windows
#define LANG_NOTSUPPORTED 89

// If the original .exe segment length (%lx) is too large or too small
#define LANG_INVALIDSEGMENTLEN 24

// When increasing the segment length
#define LANG_INCREASECODELENGTH 25

// Can't write to TTDLOAD.EXE
#define LANG_WRITEERROR 26

// Installing the code loeader
#define LANG_INSTALLLOADER 27

// TTDLOAD.EXE (%s) is invalid, needs to be deleted.
#define LANG_TTDLOADINVALID 28

// TTDLOAD.EXE was verified to be correct
#define LANG_TTDLOADOK 29


//-----------------------------------------------
//   COMMAND LINE HELP (-h)
//-----------------------------------------------

// Introduction, prefixed with "TTDPATCH V<version> - "
#define LANG_COMMANDLINEHELP 30

// Text describing the switches with values.  The lines have to be shorter
// than 79 chars, excluding the "\n".  Start new lines if necessary.
#define LANG_FULLSWITCHES 31

// Referral to the docs, prefixed by "Copyright (C) 1999 by Josef Drexler.  "
#define LANG_HELPTRAILER 32


//-----------------------------------------------
//  COMMAND LINE AND CONFIG FILE PARSING
//-----------------------------------------------

// if an on/off switch has a value other than the above (%s = wrong value)
#define LANG_UNKNOWNSTATE 33

// switch is unknown.  First %c is '-' or '/' etc, 2nd is the switch char
#define LANG_UNKNOWNSWITCH 34

// cfg command %s is unknown
#define LANG_UNKNOWNCFGLINE 35

// A cfg file (%s) could not be found and is ignored.
#define LANG_CFGFILENOTFOUND 36

// Couldn't write the config file
#define LANG_CFGFILENOTWRITABLE 37

// A non-comment line is longer than 32 chars, rest ignored.
#define LANG_CFGLINETOOLONG 38

// An obsolete switch is used
#define LANG_SWITCHOBSOLETE 85


//----------------------------------------------------
//   SWITCH DISPLAY ('-v')
//----------------------------------------------------

// Introduction
#define LANG_SHOWSWITCHINTRO 39

// Char to be put between the two columns of switches.  Shouldn't be changed unless your codepage is weird.
#define LANG_SWTABLEVERCHAR 40

// Output after "New mountain handler" or "New curve handler" if set
#define LANG_SWFASTERFOR 41

// Output before the train types for the given speed increases
#define LANG_SWFASTER    42
#define LANG_SWFULLSPEED 43

// Shows the load options for ttdload.  %s is the given parameters to be passed to ttdload
#define LANG_SWSHOWLOAD 44

#define LANG_SWABORTLOAD 45


//---------------------------------------
//  STARTUP AND REPORTING
//---------------------------------------

// Incorrect magic bytes for protected mode code
#define LANG_CHECKMAGIC 46

// DOS reports no memory available
#define LANG_NOTENOUGHMEM 47

// ...to start TTD
#define LANG_TOSTARTTTD 86

// Protected mode code exceeds 32kb
#define LANG_PROTECTEDTOOLARGE 48

// Show where the code was stored, %p is the location
#define LANG_CODESTOREDAT 49

// Shown when swapping TTDPatch out
#define LANG_SWAPPING 88

// Just before running ttdload, show this.  1st %s is ttdload.exe, 2nd is the options
#define LANG_RUNTTDLOAD 50

// Error executing ttdload.  1st %s is ttdload.exe, 2nd %s is the error message from the OS
#define LANG_RUNERROR 51

// Show the result after after running, %s is one of the following strings
#define LANG_RUNRESULT 52
#define LANG_RUNRESULTOK 53
#define LANG_RUNRESULTERROR 54

//---------------------------------------------------
//   CONFIG FILE COMMENTS (for '-W')
//---------------------------------------------------

// This is the intro at the start of the config file.  No constraints on line lengths.
#define CFG_INTRO 58

// Definitions of the cfg file comments.
// All can have a place holder %s to stand for the actual setting name,
// and all but CFG_CDPATH can have a %c *after* the %s for the command
// line switch.
// They will have the "comment" char and a space prefixed.
//
#define CFG_SHIPS 59
#define CFG_CURVES 60
#define CFG_SPREAD 61
#define CFG_TRAINREFIT 62
#define CFG_SERVINT 63
#define CFG_NOINFLATION 64
#define CFG_LARGESTATIONS 65
#define CFG_MOUNTAINS 66
#define CFG_NONSTOP 67
#define CFG_REDUCEDSAVE 68
#define CFG_PLANES 69
#define CFG_LOADTIME 70
#define CFG_ROADVEHS 71
#define CFG_SIGNCHEATS 72
#define CFG_TRAINS 73
//*#define CFG_PLAYERS
#define CFG_VERBOSE 74
#define CFG_PRESIGNALS 75
#define CFG_MOREVEHICLES 76
#define CFG_MAMMOTHTRAINS 77
#define CFG_FULLLOADANY 78
#define CFG_SELECTGOODS 79
#define CFG_DEBTMAX 80
#define CFG_OFFICEFOOD 81
#define CFG_ENGINESPERSIST 82
#define CFG_CDPATH 83
// Note- CFG_CDPATH has no command line switch, so don't give %c!
#define CFG_KEEPSMALLAP 84
#define CFG_AIBOOST 55
#define CFG_LONGBRIDGES 56
#define CFG_DYNAMITE 57
#define CFG_MULTIHEAD 87
#define CFG_RVQUEUEING 91
#define CFG_LOWMEMORY 92
#define CFG_GENERALFIXES 93
#define CFG_MOREAIRPORTS 94
#define CFG_BRIBE 95


// END LANGUAGE TEXTS
// Insert ONLY language texts or comments between the BEGIN and END lines

#define CFG_COMMENT "// "

#define LANG_LASTSTRING 96	// this will be the next one to be defined
#define SWITCHNUM (lastbit+1)	// number of switch in verbose display
#define TRAINTYPENUM 4		// number of train types for -mc display

// Index numbers of the strings in language.dat file. Must be increasing.
#define LANGCODE_NAME(i) (-0x4000-(i))
#define LANGCODE_TEXT(i) (0x400+(i))
#define LANGCODE_SWITCHES(i,j) (-0x2000-((i)*2+(j)))
#define LANGCODE_HALFLINES(i) (0x1200+(i))
#define LANGCODE_TRAINTYPES(i) (0x1400+(i))
#define LANGCODE_END(i) (0x4000+(i))
#define LANGCODE_RESERVED -0x7fff	// may never be used otherwise

// Name of this language
ISEXTERN char *langname;

// All output lines that aren't in one of the other arrays, for
// all languages
ISEXTERN char *langtext[LANG_LASTSTRING+1];

// Lines of help for all on/off switches, each at most 38 chars long.
// If you need more chars just insert another line.
ISEXTERN char **halflines;
ISEXTERN int numhalflines;

// Names of the switches for the '-v' options
// First string is shown always, second only if set and with the given
// value of the switch in %d.
// These lines (both parts) are limited to 36 chars, also consider how large
// the expansion of the %d can be for that switch.
ISEXTERN char *switchnames[SWITCHNUM*2];

// Names of the three train types, plus "none ", all trailed by a space
ISEXTERN char **traintypes;
ISEXTERN int numtraintypes;

ISEXTERN char *langnames[LANG_MAX];
ISEXTERN u32 curlang, nlang;

typedef void langfunc();
ISEXTERN langfunc *languages[LANG_MAX];

ISEXTERN u16 *countries;

void process_languages_1();
void process_languages_2();
void process_languages_3();

#undef ISEXTERN
#endif // _LANGUAGE_H
