//
// This file is part of TTDPatch
// Copyright (C) 1999-2001 by Josef Drexler
//
// switches.cpp: defines functions dealing with the switches
//

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifdef __BORLANDC__
#	include <conio.h>
#else
#	define getch() getc(stdin)
#	define stricmp strcasecmp
#endif

#include "error.h"

#define IS_SWITCHES_CPP
#include "switches.h"
#include "language.h"

int showswitches = 0;
int writeverfile = 0;
char ttdoptions[128] = "";
int cfgfilespecified = 0;

extern char *patchedfilename;


void givehelp()
{
  int i, total;

  printf("TTDPatch - ");
  printf(langtext[LANG_COMMANDLINEHELP], patchedfilename);

  for (total=0; halflines[total]; total++);

  for (i=0; i<total/2; i++)
	printf("%-38s  %-38s\n", halflines[i], halflines[i + (total + 1)/2]);
  if (total % 2)
	printf("%-38s \n", halflines[(total + 1)/2 - 1]);

  printf("%s%s%s",
	langtext[LANG_FULLSWITCHES],
	"Copyright (C) 1999-2001 by Josef Drexler.  ",
	langtext[LANG_HELPTRAILER]);
  exit(0);
}



void initvalues(int preferred)
{
  int i;

  if (preferred) {
	for (i=0; i<4; i++)
		flags->u.d.vehcount[i] = 240;

	for (i=0; i<2; i++)
		flags->u.d.mctype[i] = (~ 0x13) & 0xff;

	flags->u.d.vehicledatafactor = 1;
	flags->u.d.savesize = 1;
	flags->u.d.stationspread = 20;
	flags->u.d.newservint = 16000;
  } else {
	for (i=0; i<4; i++)
		flags->u.d.vehcount[i] = 80;

	for (i=0; i<2; i++)
		flags->u.d.mctype[i] = (~ 0x77) & 0xff;

	flags->u.d.vehicledatafactor = 1;
	flags->u.d.savesize = 1;
	flags->u.d.stationspread = 12;
	flags->u.d.newservint = 180;
  }
}

void allswitches(int reallyall, int swon)
{
  int last;

  if (
	reallyall
#if DEBUG
	|| 1		// for debug versions, -a turns on all switches
#endif
     )
	last = lastbit;
  else
	last = lastbitused;

  for (int i=0; i<=last; i++)
	if ( (i != saveonlysmall) || reallyall )
		if (i != lowmemory)
			setfv(i, swon);

  initvalues(swon);
}


#define OBSOLETE ((void*)-1L)

int radix[4] = { 0, 8, 10, 16 };

#define YESNO(ch, txt, comment, sw) \
	{ ch, txt, comment, sw,  0, 0, {-1, -1, -1}, NULL }

#define SPCL(ch, txt, comment, var) \
	{ ch, txt, comment, -1, -1, 2, {-1, -1, -1}, var }

#define RANGE(ch, txt, comment, sw, radix, varsize, var, low, high, default) \
	{ ch, txt, comment, sw, radix, varsize, {low, high, default}, var }

#define FLAGDATA(var) ( (void _fptr*) ( (s32) ( (char*) &(flags->u.d.var) - (char*) &(flags->u.d)) ) )

extern const switchinfo switches[] = {

	// Usage of the options:
	//
	// YESNO(cmdline, cfgfile, cfgtext, bitname)
	//
	// RANGE(cmdline, cfgfile, cfgtext, bitname, radix, varsize,
	//		  FLAGDATA(var), low, high, default)
	//
	// SPCL (cmdline, cfgfile, cfgtext, &var)
	//
	// cmdline is the char that is the command line option
	// cfgfile is the string that is the config file option name
	// cfgtext is the CFG_xxx constant that corresponds to this switch
	// bitname is the name of the switch in common.h
	// radix is 0 for autodetect, 1 for octal, 2 for decimal,
	//		3 for hexadecimal, add 4 to invert bitwise
	// varsize is 0 for unsigned char, 1 for unsigned int
	//		2 for signed int(2), 3 for signed long(4)
	// var is the flags->u.d.element for RANGE, or a pointer to
	//		an int for SPCL (or NULL which needs a handler)
	// low,high,default give the range and default for RANGE


		// lowercase
	SPCL ('a', "all", 0, NULL),
	RANGE('b', "ships", CFG_SHIPS, increaseshipcount, 0, 0, FLAGDATA(vehcount[3]), 1, 240, 240),
	RANGE('c', "curves", CFG_CURVES, usenewcurves, 7, 16, FLAGDATA(mctype[0]), 0, 0xff, 0x13),
	     //d
	RANGE('e', "spread", CFG_SPREAD, largerstations, 0, 0, FLAGDATA(stationspread), 1, 255, 20),
	YESNO('f', "trainrefit", CFG_TRAINREFIT, allowtrainrefit),
	YESNO('g', "generalfixes", CFG_GENERALFIXES, generalfixes),
	SPCL ('h', NULL, 0, NULL),
	RANGE('i', "servint", CFG_SERVINT, setnewservinterval, 0, 1, FLAGDATA(newservint), 1, 32767, 16000),
	     //j
	YESNO('k', "keepsmallairport", CFG_KEEPSMALLAP, keepsmallairports),
	YESNO('l', "largestations", CFG_LARGESTATIONS, morestationtracks),
	RANGE('m', "mountains", CFG_MOUNTAINS, usenewmountain, 7, 16, FLAGDATA(mctype[1]), 0, 0xff, 0x13),
	YESNO('n', "nonstop", CFG_NONSTOP, usenewnonstop),
	RANGE('o', "reducedsave", CFG_REDUCEDSAVE, saveonlysmall, 0, 0, FLAGDATA(savesize), 1, 40, 1),
	RANGE('p', "planes", CFG_PLANES, increaseplanecount, 0, 0, FLAGDATA(vehcount[2]), 1, 240, 240),
	YESNO('q', "loadtime", CFG_LOADTIME, improvedloadtimes),
	RANGE('r', "roadvehs", CFG_ROADVEHS, increaservcount, 0, 0, FLAGDATA(vehcount[1]), 1, 240, 240),
	YESNO('s', "signcheats", CFG_SIGNCHEATS, usesigncheat),
	RANGE('t', "trains", CFG_TRAINS, increasetraincount, 0, 0, FLAGDATA(vehcount[0]), 1, 240, 240),
	     //u is now obsolete, wait till ca. V1.8 for being reusable
//	SPCL ('u', "players", 0, OBSOLETE),
	SPCL ('v', "verbose", CFG_VERBOSE, &showswitches),
	YESNO('w', "presignals", CFG_PRESIGNALS, presignals),
	RANGE('x', "morevehicles", CFG_MOREVEHICLES, uselargerarray, 0, 0, FLAGDATA(vehicledatafactor), 0, 40, 1),
	SPCL ('y', NULL, 0, &alwaysrun),
	YESNO('z', "mammothtrains", CFG_MAMMOTHTRAINS, mammothtrains),
	SPCL ('?', NULL, 0, NULL),

		// uppercase
	RANGE('A', "aiboost", CFG_AIBOOST, aibooster, 0, 0, FLAGDATA(aiboostfactor), 0, 255, 0),
	YESNO('B', "longbridges", CFG_LONGBRIDGES, longerbridges),
	SPCL ('C', "include", 0, NULL),
	YESNO('D', "extradynamite", CFG_DYNAMITE, morethingsremovable),
	     //E
	YESNO('F', "fullloadany", CFG_FULLLOADANY, fullloadany),
	YESNO('G', "selectgoods", CFG_SELECTGOODS, selectstationgoods),
//	YESNO('H', "hugeairport", CFG_HUGEAIRPORT, hugeairport),
	YESNO('I', "noinflation", CFG_NOINFLATION, noinflation),
	YESNO('J', "moreairports", CFG_MOREAIRPORTS, moreairports),
	     //K
	YESNO('L', "debtmax", CFG_DEBTMAX, maxloanwithctrl),
//	YESNO('M', "moresignals", CFG_MORESIGNALS, moresignals),
	YESNO('M', "multihead", CFG_MULTIHEAD, multihead),
	     //N
	YESNO('O', "officefood", CFG_OFFICEFOOD, officefood),
	YESNO('P', "enginespersist", CFG_ENGINESPERSIST, persistentengines),
	     //Q
	YESNO('R', "rvqueueing", CFG_RVQUEUEING, newlineup),
//	YESNO('S', "fasterships"...),
//	YESNO('T', "moretrees", CFG_MORETREES, moretrees),
	     //U
	SPCL ('V', "writeverfile", 0, &writeverfile),
	SPCL ('W', "writecfg", 0, NULL),
	     //X reserved for extended switches, see below
	     //Y reserved for extended switches, see below
	YESNO('Z', "lowmemory", CFG_LOWMEMORY, lowmemory),

		// X-extended, lowercase
	YESNO('Xb',"bribe", CFG_BRIBE, bribe),
		// X-extended, uppercase
		// Y-extended, lowercase
		// Y-extended, uppercase

		// cfg-only
	SPCL (  1, "cdpath", CFG_CDPATH, NULL),

		// end-of-switches
	SPCL (  0, NULL, 0, NULL)
};

int overridesconfigfile = 0;

void readcfgfile(char *filename);
void writecfgfile(char *filename);



// Parameters for the on/off switches, case is ignored.  *MUST* be unique.
// Number of entries not limited.  End each list with a NULL entry.
// The first value is what is printed by the -W switch.
// e.g. "presignals yes" or "presignals off" etc.

char *switchonofftext[] =
	{ "on", "yes", "y", "1", NULL,
	  "off", "no", "n", "0", NULL };

char *switchofftext = switchonofftext[5];	// first switch "off" text

void _fptr *getswitchvarptr(int switchid)
{
  void _fptr *ptr = switches[switchid].var;

  if ( (u32) ptr < (u32) (void _fptr *) &error)
	ptr = (void _fptr*) ( (char _fptr*) &(flags->u.d) + (s32) ptr);

  return (void _fptr*) ptr;
}

void setswitchvar(int switchid, s32 value)
{
  void _fptr *ptr = getswitchvarptr(switchid);

  switch (switches[switchid].varsize) {
	case 0:
		*( (u8 _fptr *) ptr) = value;
		break;
	case 1:
		*( (u16 _fptr *) ptr) = value;
		break;
	case 2:
		*( (s16 _fptr *) ptr) = value;
		break;
	case 3:
		*( (s32 _fptr *) ptr) = value;
		break;
  }
}

s32 getswitchvar(int switchid)
{
  void _fptr *ptr = getswitchvarptr(switchid);

  switch (switches[switchid].varsize) {
	case 0:
		return *( (u8 _fptr *) ptr);
	case 1:
		return *( (u16 _fptr *) ptr);
	case 2:
		return *( (s16 _fptr *) ptr);
	default:
		return *( (s32 _fptr *) ptr);
  }
}

int setreallyspecial(int switchid, int swon, char *cfgpar, int onlycheck)
{
  int parused = 0;

  switch (switches[switchid].cmdline) {
	case 'a':
		if (!onlycheck) allswitches(0, swon);
		break;
	case 'h':
	case '?':
		if (!onlycheck) givehelp();
		break;
	case 'C':
		if (!onlycheck)
			readcfgfile(cfgpar);
		else
			overridesconfigfile = 1;
		parused = 1;
		break;
	case 'W':
		if (!onlycheck)
			writecfgfile(cfgpar);
		parused = 1;
		break;
	case 1:		// CD Path
		if (!onlycheck)
			if (cfgpar && (strlen(cfgpar) > 0))
				strcpy(ttdoptions, cfgpar);
			else
				strcpy(ttdoptions, "");
		parused = 1;
		break;
  }
  return parused;
}

int setswitch(int switchid, char *cfgpar, int swon, int onlycheck)
{
  int parused = 0;
  char *endptr;
  s32 parvalue;

  if (switches[switchid].bit == -1) {	// Special switches
	if (switches[switchid].var == NULL) {	// really special
		parused = setreallyspecial(switchid, swon, cfgpar, onlycheck);
	} else if (switches[switchid].var == OBSOLETE) {
			// obsolete
		if (!onlycheck) printf(langtext[LANG_SWITCHOBSOLETE],
			switches[switchid].cfgcmd, switches[switchid].cmdline);
	} else {	// with a variable to set
		if (!onlycheck)
			setswitchvar(switchid, swon);
	}
  } else if (!onlycheck) {	// not special
	setfv(switches[switchid].bit, swon);
	if ( (switches[switchid].range[0] != -1) ||
	     (switches[switchid].range[1] != -1) ) {	// with a range (value)
		if (cfgpar == NULL)
			parvalue = switches[switchid].range[2];	//default
		else
			parvalue = strtol(cfgpar, &endptr,
					radix[switches[switchid].radix & 3]);

		if ( (parvalue != 0) || ( (*endptr == 0) && (*cfgpar != 0) ) ) {
			parused = 1;

			if ( (cfgpar == NULL) ||
			     (parvalue < switches[switchid].range[0]) ||
			     (parvalue > switches[switchid].range[1]) )
				parvalue = switches[switchid].range[2];

			if (switches[switchid].radix & 4)
				parvalue = (~parvalue & (0xffffffff >> 8 * (3 - (switches[switchid].varsize + 1) / 2) ) );

			setswitchvar(switchid, parvalue);
		}
	}
  }
  return parused;
}

// command line is parsed twice, first to only check if there is a '-C' switch
// second pass is for real
int processswitch(int switchchar, char *cfgline, int swon, int onlycheck)
{

  int i, k, l, switchid = -1;
  char *cfgcmd, *cfgpar;

  if ( (switchchar == 0) && (cfgline != NULL) ) {	// cfg entry

	// syntax is "name *[:=]? *value", value may be omitted.

	i = strcspn(cfgline, " :=");
	cfgline[i] = 0;
	cfgcmd = cfgline;

	i += strspn(&(cfgline[++i]), " :=");
	cfgpar = &(cfgline[i]);

	if (*cfgpar == 0)	// no parameter
		swon = 1;	// -> "yes" default
	else {
		swon = -1;	// not yet determined
		l = 1;		// first list is "on" list

		for (k=0; l >= 0; k++)
			if (switchonofftext[k]) {
				if (stricmp(cfgpar, switchonofftext[k]) == 0) {
					swon = l;
					break;
				}
			} else {
				if (l--) switchofftext = switchonofftext[k + 1];
			}

		if (swon < 0)
			swon = 1;	// switch given, implies "on"
	}
  } else {
	cfgpar = cfgline;	// parameter
	cfgline = NULL;
	cfgcmd = NULL;
  }

  for (i=0; switches[i].cmdline; i++) {
	if (cfgline == NULL) {
		if (switches[i].cmdline == switchchar) {
			switchid = i;
			break;
		}
	} else {
		if (switches[i].cfgcmd)
			if (stricmp(cfgcmd, switches[i].cfgcmd) == 0) {
				switchid = i;
				break;
			}
	}
  }

  if (switchid == -1)
	return 0;

  return 1 + setswitch(switchid, cfgpar, swon, onlycheck);
}

void readcfgfile(char *filename)
{
  if (strlen(filename) < 1) {
	printf(langtext[LANG_CFGFILENOTFOUND], filename);
	return;
  }

  FILE *cfgfile = fopen(filename, "rt");
  if (!cfgfile) {
	printf(langtext[LANG_CFGFILENOTFOUND], filename);
	return;
  }

#define CFGLINEMAXLEN 32
  char cfgline[CFGLINEMAXLEN + 2], cfglineorg[CFGLINEMAXLEN + 2];
  int linetoolong, linepos;

  while (!feof(cfgfile)) {
	memset(cfgline, 0, CFGLINEMAXLEN + 1);
	fgets(cfgline, CFGLINEMAXLEN, cfgfile);

	linetoolong = (strchr(cfgline, '\n') == NULL) && (!feof(cfgfile));

	if (linetoolong)
		fscanf(cfgfile, "%*[^\n]");	// skip to end of line

	for (linepos = 0; isspace(cfgline[linepos]); linepos++);

	if (cfgline[linepos] == 0)
		continue;	// skip empty lines

	if (isalpha(cfgline[linepos])) {	// all lines starting with a-z are options
		if (linetoolong)
			printf(langtext[LANG_CFGLINETOOLONG]);

		strcpy(cfglineorg, cfgline);
		char *eol = strchr(cfgline, '\n');
		if (eol) *eol = 0;

		if (!processswitch(0, &(cfgline[linepos]), 0, 0))
			error(langtext[LANG_UNKNOWNCFGLINE], cfglineorg);
	}
  }

  fclose(cfgfile);
}

void writereallyspecial(int switchid, char **formatstring, s32 *parvalue)
{
  switch (switches[switchid].cmdline) {
	case 1:		// CD Path
		if (strlen(ttdoptions) > 0)
			*parvalue = (s32) ttdoptions;
		else
			*parvalue = (s32) "";
		*formatstring = "%s %s";
		break;
  }
}

void writerangedswitch(int switchid, char **formatstring, s32 *parvalue)
{
  *formatstring = tempstr;
  (*formatstring)[0] = 0;

  if (!getf(switches[switchid].bit)) {
	*formatstring = "%s %s";
	*parvalue = (s32) switchofftext;
  } else {
	switch (switches[switchid].radix & 3) {
		case 1:
			strcat(*formatstring, "%s %lo");
			break;
		case 3:
			strcat(*formatstring, "%s %lx");
			break;
		default:
			strcat(*formatstring, "%s %ld");
			break;
	}
	*parvalue = getswitchvar(switchid);
	if (switches[switchid].radix & 4)
		*parvalue = (~(*parvalue) & (0xffffffff >> 8 * (3 - (switches[switchid].varsize + 1) / 2) ) );
  }
}

void writeswitch(FILE *cfgfile, int switchid)
{
  s32 parvalue;
  char *formatstring = "%s %ld";

  if (switches[switchid].bit == -1) {		// special
	if (switches[switchid].var == NULL) {		// really special
		writereallyspecial(switchid, &formatstring, &parvalue);
	} else if (switches[switchid].var == OBSOLETE) {	// obsolete
		return;
	} else {	// has a var
		parvalue = *( (int*) switches[switchid].var);
		if ( (parvalue >= 0) && (parvalue <= 1) ) {
			formatstring = "%s %s";
			if (parvalue)
				parvalue = (s32) switchonofftext[0];
			else
				parvalue = (s32) switchofftext;
		}
	}
  } else {	// not special
	if ( (switches[switchid].range[0] != -1) ||
	     (switches[switchid].range[1] != -1) ) {	// ranged (value)
		writerangedswitch(switchid, &formatstring, &parvalue);
	} else {	// just on/off
		formatstring = "%s %s";
		if (getf(switches[switchid].bit))
			parvalue = (s32) switchonofftext[0];
		else
			parvalue = (s32) switchofftext;
	}
  }

  fprintf(cfgfile, formatstring, switches[switchid].cfgcmd, parvalue);
}

void writecfgfile(char *filename)
{
  if (strlen(filename) < 1) {
	printf(langtext[LANG_CFGFILENOTFOUND], filename);
	return;
  }

  FILE *cfgfile = fopen(filename, "wt");
  if (!cfgfile) {
	printf(langtext[LANG_CFGFILENOTWRITABLE], filename);
	return;
  }

  fprintf(cfgfile, langtext[CFG_INTRO]);

  for (int switchid=0; switches[switchid].cmdline; switchid++)
	if (switches[switchid].comment) {
		fprintf(cfgfile, "\n\n" CFG_COMMENT);
		fprintf(cfgfile, langtext[switches[switchid].comment],
			switches[switchid].cfgcmd, switches[switchid].cmdline);
		fprintf(cfgfile, "\n");

		writeswitch(cfgfile, switchid);
	}

  fprintf(cfgfile, "\n");
  fclose(cfgfile);
}

static char dcharstr[4];
char *dchartostr(int ch)
{
  *(long int *) dcharstr = (long int) ch;
  return dcharstr;
}

void commandline(int argc, char **argv)
{
  int i, par, swon, p, parused;
  char *s, *switchpar;
  int switchchar;
  int onlycheck;

  for (i=0; i<nparlong; i++)
	flags->u.longs[i] = 0;

  for (i=0; i<nflags; i++)
	flags->flags[i] = 0;

  alwaysrun = 0;

  for (onlycheck = 1; onlycheck >= 0; onlycheck--) {

	// "real" run, and there is no '-C' switch -> source default .cfg file
     if ( (!onlycheck) && (!overridesconfigfile) )
	readcfgfile("ttdpatch.cfg");

     for (par=1; par < argc; par++) {
	s = argv[par];

	switchpar = NULL;

	if ( (s[0] != '-') && (s[0] != '/') ) {
		if (!onlycheck) {	// handle non-option strings, but only in the "real" run
			if (strlen(ttdoptions) > 0)
				strcat(ttdoptions, " ");
			strcat(ttdoptions, s);
		}
		continue;
	}

	if (par+1 < argc) {
		switchpar = argv[par + 1];
		parused = 0;
	} else
		parused = -1;

	for (p=1; s[p]; p++) {
		switchchar = s[p];
		if ( (switchchar == 'X') || (switchchar == 'Y') ) {
			p++;
			if (!s[p])
				error(langtext[LANG_UNKNOWNSWITCH], s[0], dchartostr(switchchar));
			switchchar |= s[p] << 8;
		}

		if (onlycheck && (switchchar != 'C') )
			continue;

		swon = 1;
		if (s[p + 1] == '-') {
			swon = 0;
			p++;
		}

		i = processswitch(switchchar, switchpar, swon, onlycheck);
		if (!i)
			error(langtext[LANG_UNKNOWNSWITCH], s[0], dchartostr(switchchar));

		if (i == 2)
			parused = 1;
	}

	if (parused)
		par++;
     }
  }

#if !WINTTDX && !LINTTDX
  if (getf(lowmemory) && (flags->u.d.vehicledatafactor > 2))
	flags->u.d.vehicledatafactor = 2;
#endif

  if (flags->u.d.vehicledatafactor <= 1)
	clearf(uselargerarray);

  if (!getf(uselargerarray)) {
	clearf(saveonlysmall);
	flags->u.d.vehicledatafactor = 1;
  }

  if (flags->u.d.savesize >= flags->u.d.vehicledatafactor)
	clearf(saveonlysmall);

  if (!getf(saveonlysmall))
	flags->u.d.savesize = flags->u.d.vehicledatafactor;
}

	      // which variable is associated with them
	      // -1=none, 0-0xff=u8, 0x100-0x1ff=u16,
	      //  0x200-0x2ff=s16, 0x300-0x3ff=s32

int switchvalues[lastbit + 1] =
	{  4, -1, -1, -1,  8,  9, 0x100, -1,		// 0..7
	  -1, 10, 11,  5, -1,  3, -1, -1,		// 8..15
	  -1, -1, -1, -1, -1, -1, -1, -1,		// 16..23
	  -1, -1,  2, -1, -1, -1, -1, -1,		// 24..31
	  -1,						// 32..32
	};

	      // how many to show
const int shownswitches = lastbit+1 - 2;	// mountains&curves not here
const int lines = (shownswitches + 1) / 2;

	// fill with -1 if it's an even number at the end
int switchorder[lines*2] =				// display index
	{  3,  6,  7,  8, 13, 14,  0, 11,		// 0..7
	  16, 17, 18,  4,  5,  9, 10, 19,		// 8..15
	  20, 21, 22, 23, 24, 12, 15, 25,		// 16..23
	  26, 27, 28, 29, 30, 31, 32, -1,		// 24..31
	};

void showoneswitch(int bit, char *midtext, s32 value, int column)
{
  char flag = getf(bit);
  char line[40], *lineend = line;

  if (switchvalues[bit] != -1) {
	int size=switchvalues[bit] >> 8;
	int num =switchvalues[bit] & 0xff;

	switch (size) {
		case 0:	value = flags->u.bytes[num];
			break;
		case 1:	value = flags->u.words[num];
			break;
		case 2:	value = flags->u.ints[num];
			break;
		case 3:	value = flags->u.longs[num];
			break;
		default:printf("showoneswitch: Invalid variable number\n");
			exit(1);
	}

	midtext = switchnames[bit*2+1];
  }

  lineend += sprintf(line, " %c ",
			flag?langtext[LANG_SWTABLEVERCHAR][1]
			    :langtext[LANG_SWTABLEVERCHAR][2]);

  lineend += sprintf(lineend, switchnames[bit*2]);

  if (flag && midtext)
	lineend += sprintf(lineend, midtext, value);

  printf(line);

  if (column == 1)
	printf("%*c", 40-strlen(line), langtext[LANG_SWTABLEVERCHAR][0]);
  else if ( (column == 2) && (strlen(line) < 40) )
	printf("\n");
}

void decodetrains(char *text, int value)
{
  int i;

  printf(text);

  if (value == 0)
	printf(traintypes[3]);
  else
	for (i=0; i<3; i++)
		if ( (value >> i) & 1)
			printf(traintypes[i]);
}

void showtheswitches()
{

  int i;

  printf(langtext[LANG_SHOWSWITCHINTRO]);

#if DEBUG
	// sanity checks
  int k, count;
  for (k=0; k<=lastbit; k++) {
	count = 0;
	if (!switchvalues[k])
		printf("Don't have a switchvalue[%d]\n", k);
	for (i=0; i<sizeof(switchorder)/sizeof(switchorder[0]); i++)
		if (switchorder[i] == k)
			count++;
	if ( (k != usenewmountain) && (k != usenewcurves) )
		if (count != 1)
			printf("Bit %d is displayed %d times\n", k, count);
  } 

#endif //  DEBUG

  for (i=0; i<lines; i++) {
	showoneswitch(switchorder[i], "", -1, 1);
	if (switchorder[i + lines] != -1)
		showoneswitch(switchorder[i + lines], "", -1, 2);
	else
		printf("\n");
  }

  if (flags->u.d.vehicledatafactor > 0)
    setf(uselargerarray);

  for (i=0; i<2; i++) {
	showoneswitch(usenewcurves + i, langtext[LANG_SWFASTERFOR], ~(flags->u.d.mctype[i]) & 0xff, 0);
	if (i)
		printf("   ");
	if (getf(i + usenewcurves)) {
		int fastest = flags->u.d.mctype[i] & 7;
		int faster = (flags->u.d.mctype[i] >> 4) & 7;
		faster &= ~fastest;
		decodetrains(langtext[LANG_SWFASTER], faster);
		printf("  ");
		decodetrains(langtext[LANG_SWFULLSPEED], fastest);
	}
	printf("\n");
  }

  printf("\n ");
  printf(langtext[LANG_SWSHOWLOAD], ttdoptions);

  fflush(stdout);
  if (getch() == 27)
	error(langtext[LANG_SWABORTLOAD]);
  printf("\n");
}

