#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sched.h>
#include <dirent.h>
#include <asm/ldt.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <sys/wait.h>
#include <ggi/gii.h>
#include <linux/fb.h>
#include <linux/kd.h>
#include <linux/types.h>
#include <linux/vt.h>

#define executable "TYCOON.EXE"
/* TTDPatch needs 8 mb of core memory */
#define core_size 8 * 1024 * 1024
#define banked_fb_addr 0x10000000
#define linear_fb_addr 0x11000000
#define fb_size 640*480

#include "exitcodes.h"
#include "patches.h"

/*
 * Doubles the vertical resolution of the monitor (not of the game),
 * for extra image quality on large monitors.
 */
int double_scan=1;

char *exe_ptr;
char *core_ptr;

extern void* patch_locs_us[];
extern void* patch_locs_en[];
extern void* patch_locs_de[];
extern void* patch_locs_fr[];
extern void* patch_locs_es[];

int exe_corestarts[] = {0x124f60,
			0x124fa0,
			0x125b20,
			0x127340,
			0x127520,
			0};

char* exe_names[] = {"V2.01.119 American English",
                     "V2.01.119 Brittish English",
		     "V2.01.119 German",
		     "V2.01.119 French",
		     "V2.01.119 Spanish"};

void* exe_patch_locs[] = {patch_locs_us,
			  patch_locs_en,
			  patch_locs_de,
			  patch_locs_fr,
			  patch_locs_es};

void* exe_varedxxx_offsets[] = {0,0,48,48,32};
void* exe_varf9xxx_offsets[] = {0,80,1194,3444,3056};

/* Array pointing to the patch locations of the current executable */
void **patch_locs;
int varedxxx_offset,varf9xxx_offset;

FILE *logfile;

short int realmode_memblock_counter = 0;
void *realmode_memblock_pointers[256];
int realmode_memblock_sizes[256];

/*int mouse_x,mouse_y,
    last_mouse_x,last_mouse_y,
    mouse_x_min,mouse_y_min,
    mouse_x_max,mouse_y_max,
    mouse_buttons = 0*/;
    
__s16	mouse_move_x,mouse_move_y,
	mouse_buttons=0;
    
gii_input_t mouse;
gii_input_t keyboard;
gii_input_t inputs;

int fbdev_fd;
int vt_fd;

pid_t timer_pid=0;
pid_t mouse_pid=0;
pid_t game_pid=0;

int timer_thread_ready=0;
int timer_active=0;
int graphics_mode=0;
int vt_switched_away=0;

__u16 oldpal_red[256];
__u16 oldpal_green[256];
__u16 oldpal_blue[256];
__u16 gamepal_red[256];
__u16 gamepal_green[256];
__u16 gamepal_blue[256];

struct fb_var_screeninfo old_video_mode;
struct fb_cmap old_palette={0,256,oldpal_red,oldpal_green,oldpal_blue,NULL};
struct fb_cmap game_palette={0,256,gamepal_red,gamepal_green,gamepal_blue,NULL};

__u16 scrollh;
__u16 scrollv;

/* Variable needed for gettimeofday. */
struct timeval tv_prev;
int timer_val,ints;

char *save_fb;

typedef struct {
    __u16 len;
    __u8 val;
} exe_rect;

void parse_cmd_line(char **argv,int argc) {
    unsigned char *x;
    unsigned int v;

    argv++;
    while (argc>1) {
	if (strcmp(*argv,"-L")==0) {
	    argv++;
	    sscanf(*argv,"%x",&v);
	    x=(char*) 0x9ef28;
	    *x=v && 0xff;
	} else {
	    x=(char*) 0x9ee13;
	    strcpy(x,*argv);
	    while (*x!=0)
		x++;
	    if (x[-1]!='/') {
		x[0]='/';
		x[1]=0;
	    };
	};
	argv++;
    };
};

extern void* core_start;

extern bank_switch_call;
extern getcwd_call;
extern video_detect();
extern video_patch_1();
extern video_patch_2();
extern palette_patch_1();
extern palette_patch_2();
extern palette_patch_3();
extern palette_patch_4();
extern mouse_patch_1();
extern mouse_patch_2();
extern keyb_patch_1();
extern search_scenarios();
extern file_dlg_search();
extern seed_random_1();
extern seed_random_2();
extern idle_loop();
extern novell_detect();
extern open_ipx_socket();
extern send_packet();
extern receive_packet();
extern close_ipx_socket();
extern serial_detect();
extern open_serial_port();
extern send_serial_byte();
extern read_serial_byte();
extern close_serial_port();
extern char **stringtable;
extern int stringtable_size;
#ifdef USE_TTDPATCH
extern install_ttdpatch();
extern void commandline(int argc, char **argv);
#endif

void install_jmp_patch(__u32 addr, __u32 dist) {
    /* Does only support forward jumps! */
    __u8  *u;
    __u32 *v;
    
    /* Check for short jump */
    if (dist<=3) {
	u=addr;
	while (dist>0) {
	    *u=0x90;
	    u++;
	    dist--;
	}
    } else {
        if (dist-2<=0x7f) {
    	    u=(__u8 *) addr;
	    u[0]=0xeb;
	    u[1]=dist-2;
	} else {
	    u=(__u8 *) addr;
	    *u=0xe9;
	    v=(__u32 *)(addr+1);
	    *v=dist-5;
	}
    }
};

void install_jmp_patch_fixed(__u32 addr, __u32 jmpaddr) {
    install_jmp_patch(addr,(jmpaddr-addr));
}

void install_call_patch(__u32 addr,void *patch,int nops) {
    __u8 *c;
    __u32 *u;
    
    c=(char  *)addr;
    *c=0xe8;
    u=(__u32 *)(c+1);
    *u=((patch-addr)-5);
    c+=5;
    while (nops>0) {
	*c=0x90;
	c++;
	nops--;
    };
};

void do_patches() {
    __u32 *u;
    __u16 *w;

    /* Patch the core so that it no longer parses args.*/
    install_jmp_patch(patch_locs[p_arg_parse_skip],101);
    /* Sloop de seriele poort eruit.*/
//    u=0x12539e;
//    *u=0x909090c3;

    /* Disable a sound initialization routine. */
    u=patch_locs[p_sound_init_1];
    *u=0x909090c3;
    
    /* This has something to do with sound too. */
    u=patch_locs[p_sound_init_2];
    *u=0x909090c3;

    /* Skip most of the video card detection.
       Set the framebuffer address, then jump right to
       the set video mode for one of the supported cards. */
    u=patch_locs[p_video_detect_loc]+1;
    *u=((&video_detect-((int )patch_locs[p_video_detect_loc]))-5);
			
    /* Make sure our bankswitcher is installed. */
    u=patch_locs[p_bank_switch_loc];
    *u=&bank_switch_call;

    /* Disable a lock of realmode memory. (To be locked, it's loaded in es,
       crash! */
    install_jmp_patch(patch_locs[p_rmm_lock_skip],73);

    /* Disable some timer programming code. */
    install_jmp_patch(patch_locs[p_timer_prog_1_loc],28);
    install_jmp_patch(patch_locs[p_timer_prog_2_loc],34);
    
    /* Correct some detections wether a draw operation is in the video
       or the temp buffer. */
    install_call_patch(patch_locs[p_video_patch_1_loc1],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_1_loc2],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_1_loc3],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_1_loc4],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_1_loc5],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_1_loc6],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_1_loc7],&video_patch_1,6);
    install_call_patch(patch_locs[p_video_patch_2_loc1],&video_patch_2,6);
    
    /* Replace some palette programming code with our own code. */
    install_jmp_patch_fixed(patch_locs[p_palette_patch_1_loc],&palette_patch_1);
    install_jmp_patch_fixed(patch_locs[p_palette_patch_2_loc],&palette_patch_2);
    install_jmp_patch_fixed(patch_locs[p_palette_patch_3_loc],&palette_patch_3);
    install_call_patch(patch_locs[p_palette_patch_4_loc1],&palette_patch_4,22);
    install_call_patch(patch_locs[p_palette_patch_4_loc2],&palette_patch_4,22);
    
    /* Install our new getcwd. */
    install_jmp_patch_fixed(patch_locs[p_getcwd_loc],&getcwd_call);

    /* Install some mouse patches. */
    install_call_patch(patch_locs[p_mouse_patch_1_loc],&mouse_patch_1,1);
    install_call_patch(patch_locs[p_mouse_patch_2_loc],&mouse_patch_2,1);
    
    /* Install some keyboard patches. */
    install_call_patch(patch_locs[p_keyb_patch_1_loc],&keyb_patch_1,249);
    
    /* Install filesystem interface patches. */
    install_jmp_patch_fixed(patch_locs[p_search_scenarios_loc],&search_scenarios);
    install_jmp_patch_fixed(patch_locs[p_search_savegames_loc],&file_dlg_search);
    
    /* Install some new code to seed the random generator. */
    install_jmp_patch_fixed(patch_locs[p_seed_random_1_loc],&seed_random_1);
    install_jmp_patch_fixed(patch_locs[p_seed_random_2_loc],&seed_random_2);
    
    /* Modify the idle loop somewhat. */
    install_call_patch(patch_locs[p_idle_loop_loc],&idle_loop,5);
    
    /* Install some network patched. */
    install_jmp_patch_fixed(patch_locs[p_novell_detect],&novell_detect);
    install_jmp_patch_fixed(patch_locs[p_open_ipx_socket],&open_ipx_socket);
    install_jmp_patch_fixed(patch_locs[p_send_packet],&send_packet);
    install_jmp_patch_fixed(patch_locs[p_receive_packet],&receive_packet);
    install_jmp_patch_fixed(patch_locs[p_close_ipx_socket],&close_ipx_socket);
    u=patch_locs[p_network_patch_1];
    *u=0x909090c3;
    
    /* Install some serial port patches. */
    install_jmp_patch_fixed(patch_locs[p_serial_detect],&serial_detect);
    install_jmp_patch_fixed(patch_locs[p_open_serial_port],&open_serial_port);
    install_jmp_patch_fixed(patch_locs[p_send_serial_byte_1],&send_serial_byte);
    install_jmp_patch_fixed(patch_locs[p_send_serial_byte_2],&send_serial_byte);
    install_jmp_patch_fixed(patch_locs[p_read_serial_byte],&read_serial_byte);
    install_jmp_patch_fixed(patch_locs[p_close_serial_port],&close_serial_port);
    
    /* Sloopcode! */
//    w=0x128126;
//    *w=0xffcd;
//    w=0x12d334;
//    *w=0xffcd;

    /* 
     *  Install the new stringtable. The stringtable is located at
     *  $f2632
     */
//    memcpy(0xf2632,&stringtable,stringtable_size * sizeof(char*));

#ifdef USE_TTDPATCH
    install_ttdpatch();
#endif
};

#define phar_lap_p3_start	0x1980

void decompress_exe(int skip_bytes) {
    int bytes_read,bytes_written,i;
    char *p;
    char *q;
    int *core_location, *bytes_to_read;
    exe_rect er;

    core_location=exe_ptr+phar_lap_p3_start+0x26;
    bytes_to_read=core_location + 1;
    
    p=exe_ptr+phar_lap_p3_start + *core_location;
    q=core_ptr;

    bytes_read=0;
    bytes_written=0;

    while (bytes_read<*bytes_to_read) {
        er.len=*((__u16 *)p)++;
	er.val=*p++;
	bytes_read+=3;
	fprintf(logfile,"decompress: 3 bytes read, len=%u, val=%u\n",er.len,er.val);
	if ((er.len & 0x8000)==0) {
	    bytes_written+=er.len;
	    er.len--;
	    fprintf(logfile,"decompress: %u bytes read\n", er.len);
	    *q=er.val;
	    q++;
	    for (i=0; i<er.len;i++) {
		*q=*p;
		p++;
		q++;
	    };
	    bytes_read+=er.len;
	} else {
	    er.len = er.len & 0x7fff;
	    if (er.val==0) {
		/* We do not output zeros. That is because the mmaped area
		   is filled with zeros. It is allocated on access so writing
                   zero's only costs memory if a full page consists of zeros.
		while (er.len>0) {
		*q=0;
		q++;
		er.len--;
		}*/
		q+=er.len;	// Increase q instead.
		bytes_written+=er.len;
	    } else {
		while (er.len>0) {
		    memcpy(q,p,er.val);
		    bytes_written+=er.val;
		    q+=er.val;
		    er.len--;
		};
		bytes_read+=er.val;
		p+=er.val;
	        fprintf(logfile,"decompress: %u bytes read\n", er.val);
	    };
	};
    };
    fprintf(logfile,"decompress: %d bytes written\n",bytes_written);

    // For some not yet known reason there are 185820 zero's too much before the
    // core starts...
    memmove(core_ptr+0x80000,core_ptr+0x80000+skip_bytes,bytes_written-skip_bytes-0x80000);

    // Clear memory that was overlapping
    memset(core_ptr+bytes_written-skip_bytes,0,skip_bytes);

#if 0
    {
    int fd=open("tycoon.img",O_WRONLY|O_CREAT|O_TRUNC);
    write(fd,core_ptr,bytes_written-skip_bytes);
    close(fd);
    }
    exit(0);
#endif
};

#ifndef USE_IMAGE
void loadexe() {
    int fd;
    int len;
    struct stat statbuf;
    int i;
    __u32 *u;

    fprintf(logfile,"Start decompressing executable\n");
    fd=open(executable,O_RDONLY);
    if (fd<0) {
	perror("open error");
	exit(ex_open);
    };
    if (fstat(fd,&statbuf)<0) {
	perror("stat error");
	exit(ex_stat);
    };
    len=statbuf.st_size;
    exe_ptr=mmap(0,len,PROT_READ,MAP_SHARED,fd,0);
    if (exe_ptr==-1) {
	perror("mmap error");
	exit(ex_mmap);
    };
    core_ptr=mmap(0, core_size, PROT_READ|PROT_WRITE|PROT_EXEC,MAP_FIXED|MAP_PRIVATE|MAP_ANON,0,0);
    if (core_ptr==-1) {
	perror("mmap error");
	exit(ex_mmap);
    };

    u=exe_ptr+phar_lap_p3_start+0x68;
    core_start=*u;
    i=0;
    while ((exe_corestarts[i]!=0) && (exe_corestarts[i]!=core_start))
	i++;
    if (exe_corestarts[i]==0) {
	fprintf(logfile,"Your executable is unknown to loadtycoon.\n" );
	fprintf(stderr,"Your executable is unknown to loadtycoon.\n" );
	fprintf(stderr,"This means you have either an unknown version of the game,\n");
	fprintf(stderr,"or you are using a game translated into a language this loader does\n");
	fprintf(stderr,"not yet know of.\n\n");
	exit(ex_executable);
    } else {
	fprintf(logfile,"Executable recognized: %s\n",exe_names[i]);
	patch_locs=exe_patch_locs[i];
	varedxxx_offset=exe_varedxxx_offsets[i];
	varf9xxx_offset=exe_varf9xxx_offsets[i];
    };
    decompress_exe(185820);
    close(fd);

    munmap(exe_ptr,len);
};
#endif

void unload_exe() {
    munmap(core_ptr,core_size);
};

__u16 alloc_segment(void *addr,int size) {
    struct modify_ldt_ldt_s ldt;

    /* Now allocate the segment */
    realmode_memblock_pointers[realmode_memblock_counter]=addr;
    realmode_memblock_sizes[realmode_memblock_counter]=size;
    ldt.entry_number=realmode_memblock_counter+32;
    ldt.base_addr=(__u32)addr;
    ldt.limit=(size+0xfff)>>12;
    ldt.seg_32bit=1;
    ldt.contents=MODIFY_LDT_CONTENTS_DATA;
    ldt.read_exec_only=0;
    ldt.limit_in_pages=1;
    ldt.seg_not_present=0;
    ldt.useable=1;
    if (modify_ldt(1,&ldt,sizeof(ldt))!=0) {
        perror("Segment allocation error");
	exit(ex_segalloc);
    };
    realmode_memblock_counter++;
    /* To get from a descriptor number to a segment, a shift left 3 or 7
       is necessary. Our descriptors start at 32. */
    return (ldt.entry_number<<3)+7;
};


void allow_ioports() {
//    ioperm(0x3c8 ,2 ,1);  //Allow palette programming.
    if (ioperm(0x3da, 1, 1) < 0) {  //Allow vertical retrace detection
	perror("I/O permissions request failed");
	fprintf(stderr, "You have to run loadtycoon as root or make it setuid.\n");
	fprintf(stderr, "Please refer to README.\n");
	exit(ex_ioperm);
    }
};

void vthandler(int sig) {
    sigset_t newset, oldset;
    
    fprintf(logfile,"Vt-handler called\n");
    /* Block all signals while executing the handler. */
    sigfillset(&newset);
    sigprocmask(SIG_BLOCK,&newset,&oldset);
    if (vt_switched_away) {
	fprintf(logfile,"Vt-switch back to TTD vt.\n");
	/* Confirm the switch back. */
	ioctl(vt_fd, VT_RELDISP, VT_ACKACQ);
	vt_switched_away=0;
	/* Prevent a flood of interrupts. */
	gettimeofday(&tv_prev,NULL);
	/* Resume TTD */
	kill(game_pid,SIGCONT);
	kill(mouse_pid,SIGCONT);
	kill(timer_pid,SIGCONT);
	if (graphics_mode) {
	    fprintf(logfile,"Restoring display.\n");
	    if (save_fb!=NULL) {
		memcpy((char *)linear_fb_addr,save_fb,fb_size);
		munmap(save_fb,fb_size);
		save_fb=NULL;
	    };
	};
    } else {
	fprintf(logfile,"Vt-switch away from TTD vt.\n");
	/* Stop TTD while on another VT. */
	kill(game_pid,SIGSTOP);
	kill(mouse_pid,SIGSTOP);
	kill(timer_pid,SIGSTOP);
	if (graphics_mode) {
	    fprintf(logfile,"Saving display.\n");
	    save_fb=mmap(0,fb_size,PROT_READ|PROT_WRITE,MAP_PRIVATE|MAP_ANON,
	     0,0);
	    memcpy(save_fb,(char *)linear_fb_addr,fb_size);
	};
	/* Confirm the switch away. */
	ioctl(vt_fd, VT_RELDISP);
	vt_switched_away=1;
    };
    /* Enable signals again. */
    sigprocmask(SIG_BLOCK,&oldset,NULL);
};

void setup_vthandler() {
    int fd;
    struct vt_stat vtstate;
    struct vt_mode mode;
    char ttydevice[32];
    
    fd=open("/dev/tty",O_WRONLY);
    if (fd<0) {
	perror("open error");
	exit(ex_open);
    };
    if (ioctl(fd,VT_GETSTATE,&vtstate)<0) {
	perror("get vt state error");
	exit(ex_vthandler);
    };
    close(fd);
    sprintf(ttydevice,"/dev/tty%d",vtstate.v_active);
    vt_fd=open(ttydevice,O_WRONLY);
    if (vt_fd<0) {
	perror("open error");
	exit(ex_open);
    };
    if (ioctl(vt_fd,KDSETMODE,KD_GRAPHICS)<0) {
	perror("vt set graphics error");
	exit(ex_vthandler);
    };
    if (ioctl(vt_fd,VT_GETMODE,&mode)<0) {
	perror("vt-mode error");
	exit(ex_vthandler);
    };
    mode.mode=VT_PROCESS;
    mode.relsig=SIGUSR1;
    mode.acqsig=SIGUSR1;
    vt_switched_away=0;
    signal(SIGUSR1,vthandler);
    if (ioctl(vt_fd,VT_SETMODE,&mode)<0) {
	perror("vt-mode error");
	exit(ex_vthandler);
    };
};

void release_vthandler() {
    signal(SIGUSR1,SIG_DFL);
    ioctl(vt_fd,KDSETMODE,KD_TEXT);
    close(vt_fd);
};

void init_fbdev() {
    fbdev_fd=open("/dev/fb0",O_RDWR);
    if (fbdev_fd<0) {
	perror("open error");
	exit(ex_open);
    };
    /* Create a banked frame buffer */
    if (mmap((void *)banked_fb_addr,65536,PROT_READ|PROT_WRITE,MAP_FIXED|MAP_SHARED,
	 fbdev_fd,0)<0) {
	perror("mmap error");
	exit(ex_mmap);
    };
    /* Create a linear frame buffer of one megabyte.*/
    if (mmap((void *)linear_fb_addr,1024*1024,PROT_READ|PROT_WRITE,MAP_FIXED|MAP_SHARED,
	 fbdev_fd,0)<0) {
	perror("mmap error");
	exit(ex_mmap);
    };
    
};

void enter_graphics_mode() {
    struct fb_var_screeninfo si;

//    fprintf(logfile,"Entering graphics mode\n");
    /* Save the old video mode */
    if (ioctl(fbdev_fd,FBIOGET_VSCREENINFO,&old_video_mode)!=0) {
//	perror("Error getting display mode");
	exit(ex_display);
    };
    /* Save the old colour palette */
    if (ioctl(fbdev_fd,FBIOGETCMAP,&old_palette)!=0) {
//	perror("Error getting palette");
	exit(ex_display);
    };
    /*
     * From here on, "cleaner" should restore the video mode in case
     * something goes wrong.
     */
    graphics_mode=1;
    /* Set 640 x 480, 256 colours, 60 Hz */
    si.nonstd=0;
    si.xres=640;
    si.yres=480;
    si.xres_virtual=640;
    si.yres_virtual=480;
    si.xoffset=0;
    si.yoffset=0;
    si.bits_per_pixel=8;
    si.grayscale=0;
    si.activate=0;
    si.height=300;    
    si.width=400;
    si.accel_flags=0;
    si.pixclock=39722;
    if (double_scan!=0)
	si.pixclock>>=1;
    si.left_margin=48;
    si.right_margin=16;
    si.upper_margin=33;
    si.lower_margin=10;
    si.hsync_len=96;
    si.vsync_len=2;
    si.sync=0;
    if (double_scan==0)
	si.vmode=FB_VMODE_NONINTERLACED;
    else
	si.vmode=FB_VMODE_DOUBLE;
    if (ioctl(fbdev_fd,FBIOPUT_VSCREENINFO,&si)!=0) {
	perror("Error setting display mode");
	exit(ex_display);
    };    
    if (ioctl(fbdev_fd,FBIOPAN_DISPLAY,&si)!=0) {
	/* This is not a fatal error. Propably the screen is
	   already correctly panned, else the player can
	   switch to another console and back to do the trick. */
//	fprintf(logfile,"Error panning display\n");
    };
};

void leave_graphics_mode() {
    fprintf(logfile,"Leaving graphics mode\n");
    fprintf(logfile,"Length of pal: %d\n",old_palette.len);
    /* Restore the old colour palette */
    if (ioctl(fbdev_fd,FBIOPUTCMAP,&old_palette)!=0) {
	perror("Error getting palette");
	exit(ex_display);
    };
    /* Restore the old videomode */
    if (ioctl(fbdev_fd,FBIOPUT_VSCREENINFO,&old_video_mode)!=0) {
	perror("Error restoring display mode");
	exit(ex_display);
    };
    graphics_mode=0;
};
void done_fbdev() {
    munmap((void *)banked_fb_addr,65536);
    munmap((void *)banked_fb_addr,1024*1024);
    if (graphics_mode)
	leave_graphics_mode();
    close(fbdev_fd);
};

int mouse_thread(void *arg);

void input_open_error(char *msg) {
    printf("\nIt looks like you are not running loadtycoon from the console.\n");
    printf("\nHint: Running loadtycoon from Midnight Commander is *not* running\n");
    printf("      from the console.\n");
    exit(ex_ggi_open);
};

void init_input() {
    void *stack;
    struct sched_param p;

    giiInit();
    mouse=giiOpen("input-linux-mouse",NULL);
//    keyboard=giiOpen("input-stdin",NULL);
    if (mouse==NULL) {
	input_open_error("Error opening mouse\n");
    };
    keyboard=giiOpen("input-linux-kbd",NULL);
    if (keyboard==NULL) {
	input_open_error("Error opening keyboard\n");
    };
    inputs=giiJoinInputs(mouse,keyboard);
    
    /* Initialize the input queue.
    ieq_start=ieq;
    ieq_end=ieq;
    ieq_sem=0;*/
    
    fprintf(logfile,"Installing mouse thread\n");
    /* Create a stack area. Dynamically allocated. */
    stack=mmap(0,0x20000,PROT_READ|PROT_WRITE,MAP_PRIVATE|MAP_ANON,0,0);
    /* Go to end of stack. (Stack grows downwards.) */
    stack+=0x1fffc;
    
    mouse_pid=__clone(mouse_thread,stack,CLONE_VM|CLONE_FS|CLONE_FILES,NULL);
    if (mouse_pid==-1) {
	perror("Could not start mouse");
	exit(ex_clone);
    };
};

void done_input() {
    if (mouse_pid!=0) {
	kill(mouse_pid,SIGTERM);
	mouse_pid=0;
	giiClose(inputs);
	giiExit();
    };
};

/* These are markers that we return as addresses so we can detect some
   TTD actions later on. */
#define mrk_keyboard	0x01010101
#define mrk_sound	0x02020202
#define mrk_timer	0x03030303
#define mrk_mouse	0x04040404

void write_dollar_str(char *s) {
    char *t;

    t=s;
    while (*t!='$') {
	t++;
    };
    write(1,s,t-s);
};


void activate_timer() {
    fprintf(logfile, "Timer activated\n");
    /* Set the moment of the last interrupt */
    gettimeofday(&tv_prev,NULL);
    timer_val=0;
    timer_active=1;
    ints=0;
};

void deactivate_timer() {
    fprintf(logfile, "Timer deactivated, %d interrupts.\n",ints);
    timer_active=0;
};

void activate_keyboard_handler() {
    fprintf(logfile, "Keyboard activated\n");
};

void deactivate_keyboard_handler() {
    fprintf(logfile, "Keyboard deactivated\n");
};

typedef struct signal_regs {
    char unkown[20];
    __u16 gs, _gs;
    __u16 fs, _fs;
    __u16 es, _es;
    __u16 ds, _ds;
    __u32 edi;
    __u32 esi;
    __u32 ebp;
    __u32 esp;
    __u32 ebx;
    __u32 edx;
    __u32 ecx;
    __u32 eax;
    __u32 trapno;
    __u32 err;
    __u32 eip;
    __u16 cs, _cs;
    __u32 eflags;
    __u32 esp_at_signal;
    __u16 ss,_ss;
    void *fpstate;
    __u32 oldmask;
    __u32 cr2;
};

typedef struct vm86call_regs {
    short int ds;
    short int es;
    short int fs;
    short int gs;
    int eax;
    int ebx;
    int ecx;
    int edx;
};

void print_regs(struct signal_regs *sr) {
    fprintf(stderr,"eax: %08x\n",sr->eax);
    fprintf(stderr,"ebx: %08x\n",sr->ebx);
    fprintf(stderr,"ecx: %08x\n",sr->ecx);
    fprintf(stderr,"edx: %08x\n",sr->edx);
    fprintf(stderr,"esi: %08x\n",sr->esi);
    fprintf(stderr,"edi: %08x\n",sr->edi);
    fprintf(stderr,"ebp: %08x\n",sr->ebp);
    fprintf(stderr,"esp: %08x\n",sr->esp);
    fprintf(stderr,"eip: %08x\n",sr->eip);
    fprintf(stderr," ss: %04x\n",sr->ss);
    fprintf(stderr," ds: %04x\n",sr->ds);
    fprintf(stderr," es: %04x\n",sr->es);
    fprintf(stderr," fs: %04x\n",sr->fs);
    fprintf(stderr," gs: %04x\n",sr->gs);
};

void print_vm86call_regs(struct vm86call_regs *v86r) {
    fprintf(logfile,"eax: %08x\n",v86r->eax);
    fprintf(logfile,"ebx: %08x\n",v86r->ebx);
    fprintf(logfile,"ecx: %08x\n",v86r->ecx);
    fprintf(logfile,"edx: %08x\n",v86r->edx);
    fprintf(logfile," ds: %04x\n",v86r->ds);
    fprintf(logfile," es: %04x\n",v86r->es);
    fprintf(logfile," fs: %04x\n",v86r->fs);
    fprintf(logfile," gs: %04x\n",v86r->gs);
};

void print_stack(int n, int esp) {
    int *u,i;
    u=(int *)esp;
    for (i=0; i<n; i++) {
        fprintf(stderr,"stack: %x\n",*u);
	u++;
    };
};

void lprint_stack(int n, int esp) {
    int *u,i;
    u=(int *)esp;
    for (i=0; i<n; i++) {
        fprintf(logfile,"stack: %x\n",*u);
	u++;
    };
};

extern do_timer_interrupt();

int irq=0;

int timer_thread(void *arg);

void install_timer_thread() {
    void *stack;
    struct sched_param p;

    fprintf(logfile,"Installing timer thread\n");
    timer_thread_ready=0;
    /* Create a stack area. Dynamically allocated. */
    stack=mmap(0,0x20000,PROT_READ|PROT_WRITE,MAP_PRIVATE|MAP_ANON,0,0);
    /* Go to end of stack. (Stack grows downwards.) */
    stack+=0x1fffc;
    
    timer_pid=__clone(timer_thread,stack,CLONE_VM|CLONE_FS|CLONE_FILES,NULL);
    if (timer_pid==-1) {
	perror("Could not start timer_thread");
	exit(ex_clone);
    };
    /* We give the time thread a high priority. That makes it very likely
       that it gets the CPU when the nanosleep is over. This is
       good for the simulation effect; an interrupt is urgent.
       This works only when we are root. */
    p.sched_priority=1;
    if (sched_setscheduler(timer_pid,SCHED_FIFO,&p)==0) {
	fprintf(logfile,
	 "I managed to set the timer thread to maximum prority!\n");
    };
    /* In addition, we lower our priority. */
    nice(5);
};

void kill_timer_thread() {
    kill(timer_pid,SIGTERM);
    timer_pid=0;
};

void sound_driver_emulator(struct signal_regs *sr, struct vm86call_regs *v86r) {
    fprintf(logfile,"Call to sounddriver.\n");
    print_vm86call_regs(v86r);
};

typedef struct vm_statistics {
    int vmstatus;
    int nconvpg;
    int nbimpg;
    int nextpg;
    int extlim;
    int aphyspg;
    int alockpg;
    int sysphyspg;
    int nfreepg;
    int addr_start;
    int addr_end;
    int age;
    int page_fault_count;
    int swapout_count;
    int swapin_count;
    int virtual_pages_count;
    int swap_file_size; /* in pages */
    int pages_allocated_using_ems;
    int conventional_mem_pages;
    int max_swap_file_size; /* In pages */
    int page_fault_in_progress;
};

void phar_lap_call(struct signal_regs *sr) {
    unsigned char al,bh,cl;
    unsigned char *b;
    int size;
    void *ptr;
    struct vm_statistics *vmstats;

    al=(sr->eax & 0xff);
    bh=((sr->ebx>>8) & 0xff);
    cl=(sr->ecx & 0xff);
    switch (al) {
	case 0x02:
	    /* Get protected mode irq vector. */
	    switch (cl) {
		case 0x08:
		/* TTD has code to set a new irq 8 (timer) handler,
	           and restore the old one. When TTD requests the
	           address of the current handler, we return a marker
	           value. When TTD then sets the irq handler, we check
                   for this marker. If not present, activate timer
	           handler, if present, deactivate it. */
		    sr->ebx=mrk_timer;
		    break;
		case 0x09:
		/* TTD has code to set a new irq 9 (keyboard) handler,
	           and restore the old one. When TTD requests the
	           address of the current handler, we return a marker
	           value. When TTD then sets the irq handler, we check
                   for this marker. If not present, activate keyboard
	           handler, if present, deactivate it. */
		    sr->ebx=mrk_keyboard;
		    sr->es=0;
		    break;
		default:
		    fprintf(stderr,"Unsupported irq for getintvector.\n");
		    print_regs(sr);
		    exit(255);
	    };
	    break;
	case 0x03:
	    /* Get real mode irq vector. */
	    switch (cl) {
		case 0x08:
		    /* TTD has code to set a new irq 8 (timer) handler,
	               and restore the old one. When TTD requests the
	               address of the current handler, we return a marker
	               value. When TTD then sets the irq handler, we check
                       for this marker. If not present, activate timer
	               handler, if present, deactivate it. */
		    sr->ebx=mrk_timer;
		    break;
		case 0x09:
		    /* TTD has code to set a new irq 9 (keyboard) handler,
	               and restore the old one. When TTD requests the
		       address of the current handler, we return a marker
	               value. When TTD then sets the irq handler, we check
                       for this marker. If not present, activate keyboard
	               handler, if present, deactivate it. */
		    sr->ebx=mrk_keyboard;
		    break;
		case 0x33:
		    /* TTD tries to read the interrupt vector for 0x33,
		       the mouse interrupt. We return a marker again. */
		    sr->ebx=mrk_mouse;
		    break;
		case 0x66:
		    /* sound.com soundrv.com and music.com all intercept
		       int 0x66. We return a marker, so that we can detect
		       when they are called. */
		    sr->ebx=mrk_sound;
		    break;
		default:
		    fprintf(stderr,"Unsupported irq for getintvector.\n");
		    print_regs(sr);
		    exit(ex_fatal);
	    };
	    break;
	case 0x06:
	    /* Set protected mode irq handler. */
	    switch (cl) {
		case 0x08:
		    if (sr->edx==mrk_timer)
			deactivate_timer();
		    else
			activate_timer();
		    break;
		case 0x09:
		    if (sr->edx==mrk_keyboard)
			deactivate_keyboard_handler();
		    else {
			activate_keyboard_handler();
			//print_regs(sr);
			//exit(255);
		    };
		    break;
		case 0x23:
		    /* Irq 0x23 is called by Dos when ctrl-c is pressed. This
                       is Linux and we don't have any problem with ctrl-c.
                       So do nothing. */
		case 0x24:
		    /* Irq 0x24 is called by dos is bad sector/drive not ready
		       situations. Do nothing.*/
		    break;
		default:
		    fprintf(stderr,"Set irq handler not expected for irq %d.\n",cl);
		    print_regs(sr);
		    exit(ex_fatal);
	    };
	    break;
	case 0x07:
	    /* Set real & protected mode irq handler. */
	    /* TTD has code to set a new irq 9 (keyboard) handler,
	       and restore the old one. When TTD requests the
	       address of the current handler, we return a marker
	       value. When TTD then sets the irq handler, we check
               for this marker. If not present, activate keyboard
	       handler, if present, deactivate it. */
	    switch (cl) {
		case 0x08:
		    if (sr->edx==mrk_timer)
			deactivate_timer();
		    else
			activate_timer();
		    break;
		case 0x09:
		    if (sr->edx==mrk_keyboard)
			deactivate_keyboard_handler();
		    else {
			activate_keyboard_handler();
			//print_regs(sr);
			//exit(255);
		    };
		    break;
		case 0x23:
		    /* Irq 0x23 is called by Dos when ctrl-c is pressed. This
                       is Linux and we don't have any problem with ctrl-c.
                       So do nothing. */
		case 0x24:
		    /* Irq 0x24 is called by dos is bad sector/drive not ready
		       situations. Do nothing.*/
		    break;
		default:
		    fprintf(stderr,"Set irq handler not expected for irq %d.\n",cl);
		    print_regs(sr);
		    exit(ex_fatal);
	    };
	    break;
	case 0x10:
	    /* Real mode procedure call. */
	    switch (sr->ebx) {
		case mrk_sound:
		    sound_driver_emulator(sr,(struct vm86call_regs *) sr->edx);
	    	    sr->eflags=sr->eflags & ~1;
		    break;
		default:
		    fprintf(stderr,"Unknown real mode procedure call.\n");
		    print_regs(sr);
		    exit(ex_fatal);
		    break;
	    };
	    break;
	case 0x20:
	    /* Get virtual memory statistics. */
	    vmstats=sr->edx;
	    vmstats->vmstatus=1; /* Virtual memory is active */
	    /* Return dummy values to see what TTD actually uses.*/
	    vmstats->nconvpg=1001;
	    vmstats->nbimpg=1002;
	    vmstats->nextpg=1003;
	    vmstats->extlim=1004;
	    vmstats->aphyspg=1005;
	    vmstats->alockpg=1006;
	    vmstats->sysphyspg=1007;
	    vmstats->nfreepg=1008;
	    vmstats->addr_start=1009;
	    vmstats->addr_end=1010;
	    vmstats->age=0;
	    vmstats->page_fault_count=1011;
	    vmstats->swapout_count=1012;
	    vmstats->swapin_count=1013;
	    vmstats->virtual_pages_count=1014;
	    vmstats->swap_file_size=1015; /* in pages */
	    vmstats->pages_allocated_using_ems=1016;
	    vmstats->conventional_mem_pages=1017;
	    vmstats->max_swap_file_size=1018; /* In pages */
	    vmstats->page_fault_in_progress=0;
	    break;
	case 0x26:
	    /* Get information. We fill only what ttd uses. */
	    b=(char *) sr->ebx;
	    /* Guess: maybe TT is at it nicest when it is told
                      that the Dos extender uses DPMI under OS/2
		      Why OS/2? There might be special features
		      for when running under Windows. For OS/2
		      this is less likely. */
	    b[0x1c]=5; /* 5 = pentium */
	    b[0x30]=0; /* VCPI not present. */
	    b[0xd8]=1; /* DPMI present. */
	    b[0xf4]=0; /* XMS not present. */
	    b[0x11c]=0; /* Windows not present. */
	    b[0x12c]=1; /* OS/2 present. */
	    break;
	case 0x2b:
	    switch (bh) {
		case 0x05:
		    /* Lock memory region. We can ignore this, we handle
		       no irq's like under Dos. */
        	    sr->eflags=sr->eflags & ~1;
		    break;
		default:
		    fprintf(stderr,"Unsupported Phar Lap extender call.\n");
		    print_stack(6,sr->esp);
		    print_regs(sr);
		    exit(ex_fatal);
	    };
	    break;
	case 0x32:
	case 0x33:
	    /* TTD sets handlers for divide by zero and overflow
               The divide by zero handler should be able to
	       recover from a divide by zero. We have our own handler
	       for that. Overflows can be ignored. */
	    break;
	case 0xc0:
	    /* Allocate memory area in lower 1 megabytes.
	       This is going to be hard to emulate.
	       We malloc the memory and then return a handle to it.
	       The actual TTD code has to be patched to use it. */
	    size=(sr->ebx & 0xffff)<<4;
	    fprintf(logfile,"Attempt to allocate %d bytes realmode memory. ",
	     size);
	    while (realmode_memblock_pointers[realmode_memblock_counter]!=NULL)
		realmode_memblock_counter++;
	    realmode_memblock_pointers[realmode_memblock_counter]=malloc(size);
	    realmode_memblock_sizes[realmode_memblock_counter]=size;
	    sr->eax=((realmode_memblock_counter|0x100)<<3)|3;
	    fprintf(logfile,"Returned %x.\n",sr->eax);
    	    sr->eflags=sr->eflags & ~1;
	    break;
	    /* Allocate protected mode memory area.
	       We have to return a segment to the memory.
	       To do this we must allocate a selector.*/
	    size=((sr->ebx & 0xffff)<<4);
	    fprintf(logfile,"Attempt to allocate %d bytes of realmode memory. ",
	     size);
	    /* Allocate the memory using mmap */
	    ptr=mmap(0,size,PROT_READ|PROT_WRITE,MAP_PRIVATE|MAP_ANON,0,0);
	    alloc_segment(ptr,size);
	    /* Now allocate the segment */
	    fprintf(logfile,"Returned %x addr %x.\n",sr->eax,sr->ebx);
	    sr->ebx=(int)ptr;
    	    sr->eflags=sr->eflags & ~1;
	    break;
	default:
	    fprintf(stderr,"Unsupported Phar Lap extender call.\n");
	    print_stack(6,sr->esp);
	    print_regs(sr);
	    exit(ex_fatal);
    };
};

void int10_emulator(struct signal_regs *sr) {
    unsigned char ah;

    ah=((sr->eax>>8) & 0xff);
    switch (ah) {
	case 0x00:
	    if ((sr->eax & 0xff)==0x53)
		enter_graphics_mode();
	    else
		leave_graphics_mode();
	    break;
	case 0x0f:
	    /* Get videomode */
	    sr->eax=0x0000003; /* Simulate that we are in text mode. */
	    break;
	default:
            fprintf(stderr,"Unsupported int10 call!\n\n");
	    print_stack(6,sr->esp);
	    print_regs(sr);
	    exit(ex_fatal);
    };
};

void int21_emulator(struct signal_regs *sr) {
    unsigned char ah;
    unsigned char *s;
    unsigned char **t;
    unsigned int flags,size;
    void *ptr;
    struct statfs sf;

    flags=0;
    ah=((sr->eax>>8) & 0xff);
    switch (ah) {
	case 0x09:
	    /* Write dollar terminated string...*/
	    write_dollar_str((char *) sr->edx);
	    break;
	case 0x0e:
	    /* Get amount of drives. Return 1. */
	    sr->eax=(sr->eax & 0xffffff00)+1;
	    break;
	case 0x19:
	    /* Get current drive.
	       What shall we do? A:, C: ?? Let's try A: */
	    sr->eax=sr->eax & 0xffffff00;
	    break;
	case 0x1c:
	    /* Set current drive. Ignore. */
	    break;
	case 0x25:
	    /* Phar Lap dos extender call */
	    phar_lap_call(sr);
	    break;
	case 0x36:
	    /* Get free diskspace. For now emulation. It should be patched,
	       because the free diskspace can change if someone is saving his
	       game on another partition. */
	    statfs(".",&sf);
	    sr->eax=sf.f_bsize>>9;
	    if (sf.f_bfree<0x10000)
	        sr->ebx=sf.f_bfree;
	    else
		sr->ebx=0xffff;
	    sr->ecx=512;
	    if (sf.f_blocks<0x10000)
		sr->edx=sf.f_blocks;
	    else
		sr->edx=0xffff;
	    break;
	case 0x3c:
	    flags|=O_CREAT|O_TRUNC|O_RDWR;
	    goto open;
	case 0x3d:
	    /* Open file */
	    switch (sr->eax & 7) {
		case 0:
		    flags|=O_RDONLY;
		    break;
		case 1:
		    flags|=O_WRONLY;
		    break;
		case 2:
		    flags|=O_RDWR;
		    break;
	    };
	open:
	    s=(char *) sr->edx;
	    fprintf(logfile,"Opening file %s",s);
	    /* TTD is only interrested *if* something went wrong,
	       so no need for error code translation. */
	    sr->eax=open(s,flags);
	    sr->eflags=sr->eflags & ~1;
	    if ((int)sr->eax<0) {
		/* Set carry flag on error. */
		sr->eflags=sr->eflags | 1;
		fprintf(logfile," -- failed\n");
	    } else {
		fprintf(logfile," -- success handle %d\n",sr->eax);
	    };
	    break;
	case 0x3e:
	    /* Close file. */
	    fprintf(logfile,"Close file %d\n",sr->ebx & 0xffff);
	    sr->eax=close(sr->ebx & 0xffff);
	    sr->eflags=sr->eflags & ~1;
	    if ((int)sr->eax<0) {
		/* Set carry flag on error. */
		sr->eflags=sr->eflags | 1;
	    };
	    break;
	case 0x3f:
	    /* Read from file. */
	    fprintf(logfile,"Read %d bytes from file %d addr %x eip %x\n",
	     sr->ecx,sr->ebx & 0xffff,sr->edx,sr->eip);
	    s=(char *) sr->edx;
	    /* The file handle is in bx, not ebx, so "and" it first. */
	    sr->eax=read(sr->ebx & 0xffff,s,sr->ecx);
	    sr->eflags=sr->eflags & ~1;
	    if ((int)sr->eax<0) {
		/* Set carry flag on error. */
		sr->eflags=sr->eflags | 1;
		sr->eax=0;
	    };
	    break;
	case 0x40:
	    /* Write to file. */
	    fprintf(logfile,"Write %d bytes to file %d addr %x eip %x\n",
	     sr->ecx,sr->ebx & 0xffff,sr->edx,sr->eip);
	    s=(char *) sr->edx;
	    /* The file handle is in bx, not ebx, so "and" it first. */
	    sr->eax=write(sr->ebx & 0xffff,s,sr->ecx);
	    sr->eflags=sr->eflags & ~1;
	    if ((int)sr->eax<0) {
		/* Set carry flag on error. */
		sr->eflags=sr->eflags | 1;
		sr->eax=0;
	    };
	    break;
	case 0x41:
	    /* Delete file. */
	    s=(char *) sr->edx;
	    fprintf(logfile,"Deleting file %s\n",s);
	    /* TTD is only interrested *if* something went wrong,
	       so no need for error code translation. */
	    sr->eax=unlink(s);
	    sr->eflags=sr->eflags & ~1;
	    if ((int)sr->eax<0) {
		/* Set carry flag on error. */
		sr->eflags=sr->eflags | 1;
	    };
	    break;
	case 0x42:
	    /* Seek in file. */
	    switch (sr->eax & 0xff) {
		case 0x00:
		    flags=SEEK_SET;
		    break;
		case 0x01:
		    flags=SEEK_CUR;
		    break;
		case 0x02:
		    flags=SEEK_END;
		    break;
	    };
	    size=(sr->edx & 0xffff) | (sr->ecx<<16);
	    size=lseek(sr->ebx & 0xffff,size,flags);
	    fprintf(logfile,"Seek file %d, mode %d to position %x, eip %x.\n",
	     sr->ebx & 0xffff,sr->eax & 0xff,size,sr->eip);
	    sr->eax=size & 0xffff;
	    sr->edx=size>>16;
	    sr->eflags=sr->eflags & ~1;
	    if (size<0) {
		/* Set carry flag on error. */
		sr->eflags=sr->eflags | 1;
	    };
	    break;
	case 0x44:
	    /* CD-Rom drive detection? */
	    sr->ebx=1;
	    break;
	case 0x48:
	    /* Allocate protected mode memory area.
	       We have to return a segment to the memory.
	       To do this we must allocate a selector.*/
	    size=(sr->ebx & 0xffff)<<12;
	    fprintf(logfile,"Attempt to allocate %d bytes of memory. ",size);
	    /* Allocate the memory using mmap */
	    ptr=mmap(0,size,PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANON,0,0);
	    if (((int)ptr)==-1) {
        	perror("mmap error");
		print_stack(6,sr->esp);
		print_regs(sr);
		exit(ex_fatal);
	    };
	    sr->eax=alloc_segment(ptr,size);
	    fprintf(logfile,"Returned %x addr %x.\n",sr->eax,ptr);
    	    sr->eflags=sr->eflags & ~1;
	    break;
	case 0x4b:
	    /* TTD starts up the sound.com sounddrv.com and ultramid programs.
               Let's ignore it.*/
	    s=(char *) sr->edx;
	    t=(unsigned char **) (sr->ebx+6);
	    fprintf(logfile,"Attempt to execute %s with command line '%s'.\n",s,*t);
	    sr->eflags=sr->eflags & ~1;
	    break;
	case 0x4c:
	    /* Terminate TTD. */
	    exit(sr->eax & 0xff);
	default:
            fprintf(stderr,"Unsupported int21 call!\n\n");
	    print_stack(6,sr->esp);
	    print_regs(sr);
	    exit(ex_fatal);
    };
};

void int33_emulator(struct signal_regs *sr) {
    unsigned short int ax;

    ax=sr->eax & 0xffff;
    switch (ax) {
	case 0x0000:
	    /* Reset mouse driver and report number of buttons */
	    sr->eax=0xffff;
	    sr->ebx=2; /* TTD uses only 2 buttons */
	    break;
/*	case 0x0003:
	    * Get mouse position and button status *
	    sr->ebx=mouse_buttons;
	    sr->ecx=mouse_x;
	    sr->edx=mouse_y;
	    break;*/
	case 0x0004:
	    /* Set mouse position. */
	    fprintf(logfile,"Set mouse position x\n");
	    break;
	case 0x0007:
	    /* Set upper left bounds */
/*	    mouse_x_min=sr->ecx & 0xffff;
	    mouse_y_min=sr->edx & 0xffff;
	    fprintf(logfile,"Set ul bounds x: %d, y:%d\n",mouse_x_min,mouse_y_min);*/
	    break;
	case 0x0008:
	    /* Set lower right bounds */
/*	    mouse_x_max=sr->ecx & 0xffff;
	    mouse_y_max=sr->edx & 0xffff;
	    fprintf(logfile,"Set lr bounds x: %d, y:%d\n",mouse_x_max,mouse_y_max);*/
	    break;
/*	case 0x000b:
	    sr->ecx=mouse_x-last_mouse_x;
	    sr->edx=mouse_y-last_mouse_y;
	    last_mouse_x=mouse_x;
	    last_mouse_y=mouse_y;
	    break;*/
	default:
            fprintf(stderr,"Unsupported int33 call!\n\n");
	    print_regs(sr);
	    exit(ex_fatal);
    };
};

void div_by_0_intercept(int sig, siginfo_t *info, void *p) {
    struct signal_regs *sr;
    __u32 *u;

    sr=p;
    /* We only expect divide by zero in the TTD core. Therefore
       we only fix them if they happen there. */
    if (sr->eip<core_size) {
	/* Continue at the address TTD wants us to return. */
	u=0x771f2;
	sr->eip=*u;
    } else {
	fprintf(stderr,"Divide by zero outside TTD core!\n\n");
	print_stack(8,sr->esp);
	print_regs(sr);
	exit(255);
    };
};

void segv_intercept(int sig, siginfo_t *info, void *p) {
    struct signal_regs *sr;
    int i;
    unsigned char *cp;
    unsigned char intnum,portnumb;

    sr=p;
    /* We can only read the code in the TTD core. Because we only
       expect interrupts there, we only check for them there. */
    if (sr->eip<core_size) {
        cp=(char *)sr->eip;
	switch (*cp) {
	    case 0xcd:
		/* Mnemonic int executed. */
		cp++;
		/* Get int number. */
		intnum=*cp;
		cp++;
		switch (intnum) {
		    case 0x10:
			int10_emulator(sr);
			break;
		    case 0x21:
			int21_emulator(sr);
			break;
		    case 0x33:
			int33_emulator(sr);
			break;
		    default:
	    		fprintf(stderr,"This irq (%x) is not emulated!\n\n",intnum);
			print_stack(8,sr->esp);
			print_regs(sr);
			exit(ex_fatal);
		};
		sr->eip=(int) cp;
		break;
	    case 0xfa:
//		kill(timer_pid,SIGSTOP);
//		kill(mouse_pid,SIGSTOP);
		sr->eip++;
		break;
	    case 0xfb:
		/* CLI, STI. They are ignored of course. */
//		kill(timer_pid,SIGCONT);
//		kill(mouse_pid,SIGCONT);
		sr->eip++;
		break;
	    default:
		fprintf(stderr,"Trap %x! Protection fault (opcode %x)!\n\n", sr->trapno, *cp);
		fprintf(stderr,"Code: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
		 cp[0],cp[1],cp[2],cp[3],cp[4],cp[5],cp[6],cp[7],cp[8],cp[9]);
		
		print_stack(8,sr->esp);
		print_regs(sr);
		exit(ex_fatal);};
    } else {
	fprintf(stderr,"Trap %x! Protection fault! (SEGV outside TTD core)\n\n", sr->trapno);
	print_stack(8,sr->esp);
	print_regs(sr);
	exit(255);
    };
};

/*
 * This is the mouse thread. It polls the mouse and keeps track of
 * it's movement. The game gets this data in the timer thread, our
 * patches to the mouse code of the games timer handler get it.
 */

int mouse_thread(void *arg) {
    static __u8 *keypresstable = (__u8 *) 0x9c476; // Array for all keys; 0=pressed, 80h=not pressed
    gii_event event;
    __u16 key;
    __u8 *u;

#define KEY_LShift      0x2a
#define KEY_RShift      0x36
#define KEY_LCtrl       0x1d
#define KEY_RCtrl       0x9d
#define KEY_LAlt        0x38
#define KEY_RAlt        0xb8
#define KEY_CapsLock    0x3a
#define KEY_NumLock     0x45
#define KEY_ScrollLock  0x46

    while (1) {
        giiEventRead(inputs,&event,emAll);
        switch (event.any.type) {
            case evPtrButtonPress:
	        switch (event.pbutton.button) {
	            case GII_PBUTTON_FIRST:
		        mouse_buttons=mouse_buttons | 1;
		        break;
		    case GII_PBUTTON_SECOND:
		        mouse_buttons=mouse_buttons | 2;
		        break;
		};
		break;
	    case evPtrButtonRelease:
	        switch (event.pbutton.button) {
	            case GII_PBUTTON_FIRST:
		        mouse_buttons=mouse_buttons & ~1;
		        break;
		    case GII_PBUTTON_SECOND:
		        mouse_buttons=mouse_buttons & ~2;
		        break;
		};
		break;
/*	    case evPtrAbsolute:
		mouse_x=event.pmove.x;
		mouse_y=event.pmove.y;
		break;*/
	    case evPtrRelative:
	        mouse_move_x+=event.pmove.x;
	        mouse_move_y+=event.pmove.y;
	        break;
	    case evKeyPress:
	    case evKeyRepeat:
		key=event.key.sym;
		fprintf(logfile,"key 0x%x pressed: scancode 0x%x, modifiers 0x%d, label %c\n",
				key, event.key.button, event.key.modifiers, event.key.label);
		fflush(logfile);
		/*
		 * Unix Backspace/Delete hell: the second biggest design
		 * mistake ever (the PC analog joystick IO-port has
		 * the doubtfull first place) in the history of the
		 * computer.
		 * Thanks to the raw keyboard mode we use it is easier
		 * to avoid.
		 * A delete is interpreted as a backspace (erase last
		 * typed character, to avoid confusion) unless
		 * key.button reveals that it is indeed a delete.
		 */
		if (key==0x7f && event.key.button!=0x6f)
		    key=8;
		if (key==GIIK_Boot || key==3) {
		    /* On ctrl+c or ctrl+alt+del place TTD event $e0
		       which causes the quit game popup box to open. */
	            u=0x9c578;
		    *u=0xe0;
		} else if (key==7) {
		    /* Take a giant screenshot. */
		    u=0x9f15b;
		    *u=0xff;
		} else if (key==19) {
		    /* Take a normal screenshot */
		    u=0x9f15b;
		    *u=0x08;
		} else if ((event.key.modifiers & GII_MOD_ALT)==0 && key==GIIK_Up)
		    scrollv-=10;
		else if ((event.key.modifiers & GII_MOD_ALT)==0 && key==GIIK_Down)
		    scrollv+=10;
		else if ((event.key.modifiers & GII_MOD_ALT)==0 && key==GIIK_Left)
		    scrollh-=10;
		else if ((event.key.modifiers & GII_MOD_ALT)==0 && key==GIIK_Right)
		    scrollh+=10;

		/* ASCII */
		else if (key<=0x80) {
	            u=0x9c578;
		    *u=(__u8)key;

		    /* Fx */
		} else if ((event.key.button >= 0x3b && event.key.button <= 0x44)
			   || (event.key.button >= 0x57 && event.key.button <= 0x58)) {
		    int shift = !!(event.key.modifiers & GII_MOD_SHIFT);
		    int fk = event.key.button;

		    if (fk >= 0x57) fk -= (0x57 - 0x45);
		    fk -= 0x3b;

		    u=0x9c578;
		    /* Looks like TTDPatch's documentation got this reversed...? */
		    *u=(__u8)(fk + (shift ? 0xE1 : 0xF1));
		    fprintf(logfile,"sent key 0x%x\n",*u);
		}
		keypresstable[event.key.button] = 0;
		break;
	    case evKeyRelease:
		keypresstable[event.key.button] = 0x80;
		break;
	    default:
		fprintf(logfile,"Unknown event\n");
		break;
	};
    };
};

/*
 * This is the timer thread. It simulates the int 8 behaviour under Dos
 */

int timer_thread(void *arg) {
    int r;
    struct timeval tv;
    int timeelapsed;
    
    struct sigaction sigh;
    sigh.sa_sigaction=segv_intercept;
    sigemptyset(&sigh.sa_mask);
    sigh.sa_flags=SA_SIGINFO;
    sigaction(SIGSEGV, &sigh, NULL);
    /* TTD has a timer interrupt that goes off 300 times per second.
       That is a problem, because Linux has only a resolution of
       50 times per second. Therefore we ask for a resolution of
       300 times per second, but we get only 50 per second. Then
       we calculate how many interrupts we had to do in that time,
       and call them that many times.*/
    while (1) {
	irq++;
	if (timer_active==1) {
	    gettimeofday(&tv,NULL);
	    timeelapsed=tv.tv_usec-tv_prev.tv_usec;
	    tv_prev.tv_sec=tv.tv_sec;
	    tv_prev.tv_usec=tv.tv_usec;
	    if (timeelapsed<0)
		timeelapsed+=1000000;
	    timer_val+=timeelapsed;
	    kill(mouse_pid,SIGSTOP);
//	    fprintf(logfile,"%d %d\n",timer_val,timeelapsed);
	    while (timer_val>3333) {
		do_timer_interrupt();
		timer_val-=3333;
		ints++;
	    };
	    kill(mouse_pid,SIGCONT);
	};
	usleep(10000);
    };
};

/*
 * Here we install our interceptor for int instructions.
 */

void install_int_emulator() {
    struct sigaction sigh;
    sigh.sa_sigaction=segv_intercept;
    sigemptyset(&sigh.sa_mask);
    sigh.sa_flags=SA_SIGINFO;
    sigaction(SIGSEGV, &sigh, NULL);
    sigh.sa_sigaction=div_by_0_intercept;
    sigemptyset(&sigh.sa_mask);
    sigh.sa_flags=SA_SIGINFO;
    sigaction(SIGFPE, &sigh, NULL);
};

/*
 * Signal handler to clean things up in case something goes wrong
 */
 
void cleaner(int sig) {
    fprintf(logfile,"Shutdown by signal %d. Cleaning up.\n",sig);
    if (game_pid!=0)
	kill(game_pid,SIGTERM);
    done_input();
    kill_timer_thread();
    release_vthandler();
    done_fbdev();
    fclose(logfile);
    /* Reraise the signal to terminate. */
    signal(sig,SIG_DFL);
    raise(sig);
};

/*
 * Installs a signal handler to clean things up in case of a signal
 */

void install_cleaner() {
    signal(SIGINT,cleaner);
    signal(SIGQUIT,cleaner);
    signal(SIGTERM,cleaner);
    signal(SIGTRAP,cleaner);
    signal(SIGSEGV,cleaner);
    signal(SIGFPE,cleaner);
};

/*
 * This is the main thread that starts up the game!
 */

int game_thread(void *arg) {
    /* 
     * Restore some signal handlers to their default behaviour. In case
     * of an interrupt, only the main thread should do the clean up.
     */
    void *p;
    signal(SIGINT,SIG_DFL);
    signal(SIGQUIT,SIG_DFL);
    signal(SIGTERM,SIG_DFL);
    signal(SIGTRAP,SIG_DFL);
    signal(SIGSEGV,SIG_DFL);
    signal(SIGFPE,SIG_DFL);
    install_int_emulator();
    
    /* Prevent initial scrolling */
    scrollh=0;
    scrollv=0;
    /* Drop privileges (in case we are set uid root) */
    setuid(getuid());
    start_core();
};

/*
 * We run the game in a separate thread to be able to takes the right action
 * in case something goes wrong. This allows us to clean some things,
 * like restoring the frame buffer. It also allows us to pause the game
 * if someone switches to a different vt.
 */

void start_game() {
    int status;
    pid_t pid;
    void *stack;
    
    /* Create a stack area. Dynamically allocated. */
    stack=mmap(0,0x20000,PROT_READ|PROT_WRITE,MAP_PRIVATE|MAP_ANON,0,0);
    /* Go to end of stack. (Stack grows downwards.) */
    stack+=0x1fffc;
    
    allow_ioports();
    setup_vthandler();
    // If error happens in this stage, screen won't be updated if we will dump
    // errors on it. This is why we call allow_ioports() so early, not making
    // loadtycoon root is a common mistake.
    game_pid=__clone(game_thread,stack,CLONE_VM|CLONE_FS|CLONE_FILES,NULL);
    if (game_pid==-1) {
	perror("Could not start main game");
	exit(ex_clone);
    };
    pid=waitpid(-1,&status,__WCLONE);
    release_vthandler();
    if (pid==-1)
	perror("Error in waitpid");
    if (pid!=game_pid || !WIFEXITED(status)) {
	fprintf(stderr,
	 "Something unexpected happened in one of the subthreads!\n");
	fprintf(stderr,"Shutting down...\n");
    };
    if (pid==game_pid)
	game_pid=0;
    if (pid==mouse_pid)
	mouse_pid=0;
    if (pid==timer_pid)
	timer_pid=0;
};

int main(int argc, char **argv) {
    void *addr;
    DIR *d;
    void *p;
    
#ifdef USE_TTDPATCH
    parse_ttdpatch_cmdline(argc,argv);
#endif 

    logfile=fopen("loadtycoon.log","w+");
    chdir("tycoon");
#ifndef USE_IMAGE
    loadexe();
#endif
    install_cleaner();
    init_fbdev();
    /*
     * Here comes a workaround for a bug in Linux: If you use both
     * segments and threads, you must have allocated at least one
     * segment *before* starting the first thread. Otherwise your
     * segment registers will be magically set to zero at random
     * moments.
     *
     * Many thanks to Joris van Rantwijk <joris@deadlock.et.tudelft.nl>
     * for investigating this problem.
     */
    alloc_segment(0,4096);
    
    install_timer_thread();
    init_input();

#ifndef USE_TTDPATCH
    parse_cmd_line(argv,argc);
#endif

    fprintf(logfile,"Installing patches\n");
    do_patches();
    fprintf(logfile,"Starting core with fingers crossed!\n");
    start_game();
    done_input();
    kill_timer_thread();
    fclose(logfile);
};
