//
// Loading and organisation of some alternate graphics
//

#if WINTTDX
	var newgrftxt, db "newgrfw.cfg",0
#else
	var newgrftxt, db "newgrf.cfg",0
#endif

	// first sprite number to be used for TTDPatch sprites
baseoursprites equ 4900

	// GRF version we currently support
#define THISGRFVERSION 2

	// maximum GRF version compatible with this patch version
#define MAXGRFVERSION 4

uvard spriteerror	// holds pointer to spriteblock with error

	// data recorded for each entry in newgrf(w).txt
	// this is linked list starting at spriteblockptr and
	// linked via .next
struc spriteblock
	.spritelist:	resd 1	// pointer to sprite list
	.grfid:		resd 1	// GRF ID of this block
	.filenameptr:	resd 1	// Pointer to filename
	.paramptr:	resd 1	// Pointer to sprite parameters
	.next:		resd 1	// Pointer to next spriteblock
	.action8:	resd 1	// Pointer to action 8 data
	.numsprites:	resw 1	// number of sprites in this block
	.firstsprite:	resw 1	// linear number (for all blocks) of first sprite
	.cursprite:	resw 1	// and of the current sprite (for error messages)
	.numparam:	resb 1	// number of parameters given for this block
	.active:	resb 1	// 0=not active, 1=active in current game,
				// 2=will be not active, 3=will be active,
				// 80=faulty (will be skipped)
				// (2,3: is in list, but not processed yet)
endstruc_32

// called from initialize, with EDI pointing to the first byte of
// the "load GRF file" function in TTD
// Gets all necessary file and graphics functions and variables
proc initializegraphics
	local hnd

	_enter

	pusha

	storefunctiontarget 12,openfilefn
	add edi,56+3*WINTTDX
	mov eax,[edi+8]
	mov dword [curfileofsptr],eax

	storefunctiontarget 17,readwordfn
	add eax,17
	mov [copyspriteinfofn],eax
	add eax,64
	mov [readspriteinfofn],eax
	add eax,0x89
	mov [decodespritefn],eax

	// since 2.0 beta 1, ship data is *only* in newships.grf
	// so we only reserve a sprite block, but no data

	// add "fake" sprite block at the beginning to hold new ship data
	// in case the ship graphics aren't loaded (we do this even if
	// newships is off, because then they'll just be ignored later, but
	// some other things are initialized properly only if we do this)

	// allocate memory to hold the block and 1 sprite (action 8)
	push byte spriteblock_size + 4
	call spritemalloc
	pop eax
	mov [spriteblockptr],eax
	mov [curspriteblock],eax
	jc .none	// no memory

	and dword [eax+spriteblock.next],0

		// insert ship initialization sprites that don't come
		// from a .grf file

	lea edi,[eax+spriteblock_size]

	mov [eax+spriteblock.spritelist],edi
	mov byte [eax+spriteblock.numsprites],1

		// add four to skip initial dword and set bit 30
		// to mark this data as being in DS
	mov eax,newshipgrfid+4
	stosd

//	mov dword [numsprites],1+baseoursprites

	// process newgrf(w).cfg
	call processnewgrf

.none:

.fail:
	popa

	_ret
endproc initializegraphics

proc processnewgrf
	local txtbuf,txtofs,txtlen,numparam,hnd,countonly,prevspriteblock

	_enter

	mov ax,0x3d40
	mov edx,newgrftxt
	CALLINT21
	jc near .fail

	xchg eax,ebx

	// find the file length
	mov ax,0x4202
	xor ecx,ecx
	xor edx,edx

	CALLINT21
	jc .fail

	shl edx,16
	mov dx,ax
	push edx

	add edx,4
	push edx
	call spritemalloc
	pop dword [%$txtbuf]
	jc .fail	// leave clears up the stack too

	// and rewind
	mov ax,0x4200
	xor ecx,ecx
	xor edx,edx
	CALLINT21

	pop ecx		// now ecx=file size
	mov [%$txtlen],ecx
	mov edx,[%$txtbuf]
	mov [%$txtofs],edx
	jc .fail

	// read entire file
	mov ah,0x3f
	CALLINT21
	jnc .ok

.fail:
	_ret

.ok:
	and dword [edx+eax],0	// make sure there's a stop mark

	mov ah,0x3e	// close file
	CALLINT21


.nextgrf:
	and dword [%$numparam],0

	mov esi,[%$txtofs]
	and dword [%$countonly],0
	pusha
	call parseline
	popa

	jc .nextgrf		// skip if file couldn't be opened
	jz .done

	mov [%$txtofs],esi
	mov edi,[%$numparam]
	test edi,edi
	jz .noparams

	shl edi,2
	push edi
	call spritemalloc
	pop edi
	jc .fail		// leave clears up the stack too

.noparams:
	inc dword [%$countonly]
	push edi
	call parseline
	pop edi

	mov eax,[curspriteblock]
	mov [%$prevspriteblock],eax

	mov eax,[%$numparam]
	call readgrffile

#if 0
	mov eax,[numsprites]
	mov esi,[curspriteblock]
	movzx ecx,word [esi+spriteblock.numsprites]
	cmp ch,-1
	je .nextgrf

	add eax,ecx
	test ah,0xc0
	jnz .toomany

	mov [numsprites],eax
#endif
	jmp .nextgrf

#if 0
	// too many sprites
.toomany:
		// terminate linked list of sprite blocks at block
		// before this one to avoid buffer overruns
	mov eax,[%$prevspriteblock]
	and dword [eax+spriteblock.next],0

	mov ax,ourtext(toomanyspritestotal)
	call setspriteerror
#endif

.done:
	mov ebx,[%$hnd]
	_ret


// parse the newgrf.txt file pointed to by esi
//
// out:	edx = filename
//	zero = reached end of newgrf.txt
//	carry = file doesn't exist/can't be opened
parseline:
	mov bl,0	// bl = numparam

.restart:

.nextchar:
	lodsb

.tryagain:
	test al,al
	jz .foundeos

	cmp al,32
	je .foundparam
	cmp al,9
	je .foundparam

	cmp al,13
	je .foundeol
	cmp al,10
	jne .nextchar

		// found filename terminator
		// see whether know if parameters follow
.foundeos:
	cmp dword [%$numparam],0	// bl is still zero!
	ja .foundparam

.foundeol:
	test bl,bl	// write filename terminator; if we're really at the
	jnz .skipchar	// end of the filename and not the parameter list
	mov byte [esi-1],0

.skipchar:
	lodsb
	cmp al,10
	je .skipchar
	cmp al,13
	je .skipchar
	dec esi

.nextline:
	mov edx,[%$txtofs]
	mov [%$txtofs],esi
	mov [%$numparam],bl

	mov eax,edx
	sub eax,[%$txtbuf]
	sub eax,[%$txtlen]
	sbb eax,eax			// 0 if at or beyond end of data, -1 if not
	je .done

	cmp byte [edx],'#'
	stc
	je .done	// skip comments, indicate as "file not found"
			// but without error message

	// see if we can open the file
	mov ax,0x3d40
	push edx
	CALLINT21
	pop edx
	jc .fail

	// close it again
	mov bx,ax
	mov ah,0x3e

	push edx
	CALLINT21
	pop edx

	or al,1		// clear zero and carry
	jmp short .done

.fail:
	// fail with error message
	mov eax,[curspriteblock]
	mov [eax+spriteblock.filenameptr],edx	// store the file name
	mov ax,ourtext(filenotfound)
	call setspriteerror
	stc

.done:
	_ret 0

.noparam:
	mov al,13
	dec esi
	mov [esi-1],al
	jmp .tryagain

.foundparam:
	test bl,bl
	jnz .skipspace

		// write filename terminator only once, not after any parameters
	mov byte [esi-1],0

.skipspace:
	lodsb
	cmp al,9
	je .skipspace
	cmp al,' '
	je .skipspace
	jb .noparam

	sub al,'-'
	sete bh
	je .isneg

	sub al,'0' - '-'
	jb .skipparam
	cmp al,9
	ja .skipparam

.isneg:
	movzx edx,al

.nextdigit:
	xor eax,eax
	lodsb

	sub al,'0'
	jb .gotit
	cmp al,9
	ja .gotit
	imul edx,10
	add edx,eax
	loop .nextdigit

.gotit:
	test bh,bh
	jz .notneg

	neg edx

.notneg:
	cmp dword [%$countonly],0
	je .countonly

	xchg eax,edx
	stosd

.countonly:
	inc ebx

.skipparam:
	dec esi
	jmp .restart
endproc initializegraphics


// in:	edx=pointer to filename
proc readgrffile
	local sprite,spriteptr,len,numsprites,filename,numparam,paramofs

	_enter

	mov [%$numparam],eax
	mov [%$filename],edx
	mov [%$paramofs],edi
	call dword [openfilefn]
	jc near .fail		// file open failed

	mov [tempspritefilehandle],bx

	xor esi,esi

	mov [curfileblocksize],si	//0

	mov eax,dword [curfileofsptr]
	mov dword [eax],esi	//0

	or dword [%$sprite],byte -1

.nextsprite:
	call dword [readwordfn]
	cwde
	mov [%$len],eax
	test eax,eax
	jz near .done

	// waste another 4 bytes for each sprite to store various data
	// for the sprite actions

	add eax,4
	push eax
	call spritemalloc
	pop edi
	jc near .outofmem

	add edi,4

	mov esi,[%$sprite]
	cmp esi,byte -1
	je .first

	cmp esi,[%$numsprites]
	jae .toomany

	mov eax,[%$spriteptr]
	mov [eax+esi*4],edi

.first:
	mov eax,[%$len]
	mov [edi-4],eax
	call dword [copyspriteinfofn]
	call dword [decodespritefn]

	cmp dword [%$sprite],byte -1
	jne short .notspritenum

	// first sprite in the file, tells us how many sprites there are
	cmp dword [%$len],4
	jb .toomany		// must have at least four bytes of length info
				// if it has fewer it's not a valid number, therefore
				// the .grf is considered to have too many

	mov eax,[curspriteblock]

	push byte spriteblock_size
	call spritemalloc
	pop esi

	mov [eax+spriteblock.next],esi
	mov [curspriteblock],esi

	mov eax,[%$filename]
	mov [esi+spriteblock.filenameptr],eax
	mov eax,[%$paramofs]
	mov [esi+spriteblock.paramptr],eax
	mov al,[%$numparam]
	mov [esi+spriteblock.numparam],al
	and dword [esi+spriteblock.next],0

	sub edi,[%$len]		// edi-%$len = sprite data
	mov eax,[edi]		// = number of sprites
	mov [%$numsprites],eax
	shl eax,2
	push eax
	call spritemalloc
	pop eax
	mov [%$spriteptr],eax
	mov [esi+spriteblock.spritelist],eax
	jc .outofmem

.notspritenum:
	inc dword [%$sprite]
	jmp .nextsprite

.toomany:
	mov ax,ourtext(toomanysprites)
	jmp short .error

.outofmem:
	mov ax,ourtext(outofmemory)

.error:
	call setspriteerror

.done:
	mov ah,0x3e		// close file
	mov bx,[tempspritefilehandle]
	CALLINT21

	mov esi,[%$sprite]
	mov eax,[curspriteblock]
	mov [eax+spriteblock.numsprites],si

	cmp esi,1
	jl .fail

	testflags canmodifygraphics,bts

.fail:
	_ret

endproc // readgrffile

// set/record a sprite error message
//
// in:	ax=error text
// destroys eax
//
setspriteerror:
	cmp dword [spriteerror],0
	jne .alreadyhaveerror
	mov [operrormsg2],ax
	mov eax,[curspriteblock]
	mov [spriteerror],eax
.alreadyhaveerror:
	mov eax,[curspriteblock]
	mov byte [eax+spriteblock.active],0x80
	ret

#if !WINTTDX
allocspritecache:
	mov [spritecachesize],ebx
	push ebx
	call heapmalloc
	pop dword [spritecache]
	mov word [spritecacheselector],ds
	// maybe try locking the page? TTD does it...
	ret

clearspritecacheblock:
	mov byte [es:esi+4],0
	mov esi,[spritecache]
	ret
#endif

translatesprite:
	cmp bx,4890+6*WINTTDX
	jnb .notttdsprite

		// may sometimes be called with a different DS so use ss:
	mov bx,[ss: newttdsprites+ebx*2]

.notttdsprite:
	mov [ss: 0xffff+ebx*2],si
ovar storespritelastrequestnum,-4
	ret



// read and resolve all references in each sprite block
resolvesprites:
	pusha

#if 0
	// build a complete list of all offsets from each block
	// and store the sprite sizes and offsets

	mov ebx,[spriteblockptr]
	mov edi,[newspritedata]

	test ebx,ebx
	js .skip

	mov edx,baseoursprites

.nextblock:
	push ebx
	mov esi,[ebx+spriteblock.spritelist]
	movzx ecx,word [ebx+spriteblock.numsprites]
	cmp ch,-1
	je .skipblock

.next:
	lodsd
	mov [edi+edx*4],eax

		// store sprite sizes and offsets
	mov ebx,[eax-4]
	mov [edi+edx*2+16384*4],bx	// sprite size

	mov ebx,[eax+2]
	mov [edi+edx*2+16384*6],bx	// x size

	movzx ebx,byte [eax+1]
	mov [edi+edx*2+16384*8],bx	// y size

	mov ebx,[eax+4]
	mov [edi+edx*2+16384*10],bx	// x offset

	mov ebx,[eax+6]
	mov [edi+edx*2+16384*12],bx	// y offset
.pseudosprite:
	inc edx
	loop .next

.skipblock:
	pop ebx
	mov ebx,[ebx+spriteblock.next]
	test ebx,ebx
	jnz .nextblock

.skip:
#endif

	// then process each block individually, so that it can
	// adjust its sprite numbers to the real sprite numbers
	param_call procallsprites,addr(pseudospriteaction_initialize)

.done:
	popa
	ret


procallsprites:
	mov edx,[spriteblockptr]

	test edx,edx
	jns .procblock

	ret 4

.procblock:
	xor edi,edi
	mov [curspriteblock],edx

	mov word [edx+spriteblock.firstsprite],di

	cmp byte [edx+spriteblock.active],0x80
	jne .notfaulty

	add di,[edx+spriteblock.numsprites]

	jmp short .blockdone

.notfaulty:
	mov ebp,[edx+spriteblock.paramptr]
	movzx ecx,word [edx+spriteblock.numsprites]
	jecxz .blockdone

	mov eax,[edx+spriteblock.spritelist]
	test eax,eax
	jle .done

.nextsprite:
	cmp edi,dword [edx+spriteblock.numsprites]
	jnl short .done		// "less" because numsprites can be -1

	mov word [edx+spriteblock.cursprite],di

	mov esi,[eax+edi*4]
	inc edi

	test esi,esi
	jle .skipsprite		// signed or zero = bad value

.notinds:
	pusha
	call dword [esp+0x24]

	pop eax			// original EDI
	mov ecx,[esp+0x14]	// original ECX
	mov edx,[esp+0x10]	// original EDX
	sub edi,eax		// adjust by how many sprites have been skipped
	jns .dontskiprestofgrf

.skiprestofgrf:
	lea edi,[ecx-1]

	test byte [edx+spriteblock.active],2	// did we skip action 8?
	jz .dontskiprestofgrf

	// yes: mark it as inactive
	mov byte [edx+spriteblock.active],0

.dontskiprestofgrf:
	cmp ecx,edi
	ja .nottoomuch

	push eax
	mov ax,ourtext(invalidsprite)
	call setspriteerror
	pop eax
	jmp .skiprestofgrf

.nottoomuch:
	sub [esp+0x14],edi
	add eax,edi
	push eax

	popa

.skipsprite:
	loop .nextsprite

.blockdone:
	cmp dword [edx+spriteblock.grfid],0
	jne .nextblock

	cmp byte [edx+spriteblock.active],0x80	// had other errors?
	je .nextblock

	mov ax,ourtext(wronggrfversion)
	call setspriteerror

.nextblock:
	mov edx,[edx+spriteblock.next]
	test edx,edx
	jnz .procblock

.done:
	ret 4
; endp procallsprites


// do one pseudo-sprite action
// in:	eax=sprite action
//	esi=remaining pseudo-sprite data
pseudospriteaction_initialize:
	mov ebx,spriteinitializeaction
	jmp short pseudospriteaction_activate.doaction

pseudospriteaction_activate:
	mov ebx,spriteactivateaction

.doaction:
	xor eax,eax
	lodsb
	mov ebx,[ebx+eax*4]
	mov ecx,eax
	cmp al,numspriteactions
	jb .goodaction

	mov ax,ourtext(invalidsprite)
	call setspriteerror
	or edi,byte -1
	ret

.goodaction:
	test ebx,ebx
	jnle short .good

.done:
	ret

.good:
	lodsb

	cmp ecx,5
	jnb .always	// non-vehicle specific actions are always carried out

	dec ecx		// always do action 1 to skip the sprites
	jz short .always

	lea ecx,[eax+newtrains]

	testflags ecx		// all others only if enabled
	jnc short .done

.always:
	mov ecx,eax
	shl ecx,2		// lea ecx,[eax*4] is 7 bytes
	jmp ebx
; endp pseudospriteaction


	//
	// **************************************
	//
	// pseudo-sprite action handlers
	//
	// all are called with the following parameters:
	//	eax=first byte of data
	//	ecx=eax*4
	//	edx->sprite block
	//	es:esi=remainder of data
	//	edi=number of following sprite
	//	ebp=parameters from newgrf.txt
	//
	// must preserve (or adjust properly) edi
	// all other registers are fair game
	//
	// **************************************
	//


	// *** action 0 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

proc processnewinfo
	local specialfn,vehtype,infobase,infosizes,numinfo,offset,gennum,maxprop

	_enter

	push edi

	mov [%$vehtype],eax
	mov ebx,[specialpropertyfn+ecx]
	mov [%$specialfn],ebx
	mov ebx,[specificpropertybase+ecx]
	mov [%$infobase],ebx
	mov ebx,[specificpropertylist+ecx]
	mov [%$infosizes],ebx
	mov al,[maxprop+eax]
	mov [%$maxprop],eax
	mov al,[ebx]
	add al,8			// the first 8 entries are in the vehtype structure (only 7 are, actually)
	mov [%$gennum],al


	xor eax,eax
	lodsb
	mov ecx,eax	// num-props
	lodsb
	mov [%$numinfo],eax
	lodsb
	mov [%$offset],eax

	add eax,[%$numinfo]
	mov ebx,[%$vehtype]
	mov bl,[vehbnum+ebx]
	cmp eax,ebx
	jbe .nextprop

.invalid:
	mov ax,ourtext(invalidsprite)
	call setspriteerror
	pop edi
	or edi,byte -1
	_ret

.nextprop:
	xor eax,eax
	push ecx
	mov ecx,[%$numinfo]
	lodsb

	lea ebx,[eax+ecx-1]
	cmp ebx,[%$maxprop]
	ja .invalid

	cmp al,7
	jb .genprop
	je .loadamount

	cmp al,[%$gennum]
	jb .specificprop

	mov ebx,[%$offset]

	call [%$specialfn]

.next:
	pop ecx
	loop .nextprop

.done:
	pop edi

	_ret

.loadamount:
	xor ebx,ebx
	inc ebx
	mov eax,loadamount
	mov dl,1
	jmp .getgenprop

.genprop:
	// need edi=[vehtypedataptr]+(vehbase[type]+offset)*vehtypeinfo_size+eax

	mov ebx,vehtypeinfo_size
	mov dl,[gendata+1+eax]
	add eax,[vehtypedataptr]

.getgenprop:
	mov edi,[%$vehtype]
	movzx edi,byte [vehbase+edi]
	add edi,[%$offset]
	imul edi,ebx
	add edi,eax
	jmp short .doprop

.specificprop:
	sub al,8

	// need edi=infobase+al*vehbnum[type]+offset*entrysize

	mov edi,[%$infobase]

	mov edx,[%$infosizes]
	mov dl,[edx+eax+1]		// size of entry

	mov ebx,[%$vehtype]
	mul byte [vehbnum+ebx]
	add edi,eax

	mov al,dl
	and al,7			// mask out 0x80 bit
	mul byte [%$offset]
	add edi,eax

	movzx ebx,dl

.doprop:
	// now:
	// edi=first entry
	// ebx=how many bytes for each array entry
	//  dl=size (1, 2, 4, 0x84)

	cmp dl,2
	je .wordvalue		// 2
	jb .bytevalue		// 1
#if WINTTDX
	js .pointervalue	// 0x84
#endif
				// 4

.dwordvalue:
	lodsd
	mov [edi],eax
	add edi,ebx
	loop .dwordvalue
	jmp .next

.wordvalue:
	lodsw
	mov [edi],ax
	add edi,ebx
	loop .wordvalue
	jmp .next

.bytevalue:
	lodsb
	mov [edi],al
	add edi,ebx
	loop .bytevalue
	jmp .next

#if WINTTDX
.pointervalue:
	lodsd
	add eax,datastart
	mov [edi],eax
	add edi,ebx
	loop .pointervalue
	jmp .next
#endif
endproc // processnewinfo


	// *** action 1 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

newspriteblock:
	lodsb
	mov ebp,eax

		// get total number of sprites = num-sprites*num-dirs
	lodsb
	imul ebp,eax

	add ebp,edi
	cmp ebp,[edx+spriteblock.numsprites]
	ja newcargoid.invalid

	dec eax
	mov [spriteand],al

	mov edi,ebp	// skip sprites
	ret
; endp newspriteblock

activatevehspriteblock:
	lodsb
	mov ecx,eax
	lodsb
	imul ecx,eax
	call insertactivespriteblock
	mov [spritebase],eax
	ret


	// *** action 2 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - regular cargo ID:
	//   <00> <W:base-sprite> <B:and-mask>
	//	base-sprite: real sprite number of first sprite in action 1
	//		     after mapping into TTD's sprite number space
	//	and-mask: mask for the direction variable
	// - random ID:
	//   (unused)
	// - variational ID:
	//   (unused)
	//
	// Also all sprite numbers are translated into numbers relative
	// to the action 1 sprite number.

newcargoid:
	and dword [esi-6],0

	mov al,[spriteand]
	mov [esi-3],al	// store and mask in front of cargo ID data

		// find out the number of directions and store in cl
	lea ecx,[eax+1]

	lodsb			// cargo id
	lea ebx,[edi-1]		// edi is one too high
	mov [cargoids+2*eax],bx	// store sprite number for this cargo ID

	lodsb			// num-loadtypes or random/variational bit
	cmp al,0x83
	je .randomid

	cmp al,0x80
	// jb .cargoid
	je .randomid
	ja .variationalid

.cargoid:
	mov edx,eax
	lodsb
	add edx,eax

.adjustnext:
	lodsw
	mul cl
	mov [esi-2],ax

	dec edx
	jnz .adjustnext
	ret

.randomid:
	inc esi			// random-type
	inc esi			// randbit
	lodsb			// nrand

	mov ecx,eax

	dec eax
	mov [esi-1],al	// instead of nrand, store mask

.adjustrandom:
	lodsw
	test ah,ah
	jnz .invalid

	mov ax,[cargoids+2*eax]
	mov [esi-2],ax
	loop .adjustrandom
	ret

.invalid:
	mov ax,ourtext(invalidsprite)
	call setspriteerror
	or edi,byte -1
	ret

.variationalid:
	add esi,3
	lodsb			// nvar

	lea ecx,[eax+1]		// +1  because there is a default cargo ID

.nextvariation:
	lodsw
	test ah,ah
	jnz .invalid

	mov ax,[cargoids+2*eax]
	mov [esi-2],ax
	lodsw			// skip ranges
	loop .nextvariation
	ret
; endp newcargoid


activatecargoid:
	cmp byte [esi+1],0x80
	jb .realcargoid
	ret

.realcargoid:
	mov ebx,[spritebase]	// that's really a WORD value
	mov eax,ebx
	xchg ax,[esi-5]
	sub ebx,eax

	xor eax,eax
	lodsb		// cargo-id

	// now new sprite base saved at esi-5, and difference new-old is in ebx
	lodsb		// num-loadtypes
	mov ecx,eax
	lodsb
	add ecx,eax

.adjustnext:
	add [esi],bx
	inc esi
	inc esi
	loop .adjustnext
	ret


	// *** action 3 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// <B:num-veh> <W:sprite-num> <B:num-overrides>
	//	num-veh: number of vehicles in this definition, with bit 7 clear
	//	sprite-num: number of this sprite, for wagon overrides
	//	num-overrides: 0x80 + number of wagon overrides that follow or 0x00

initializevehcargomap:
	and dword [esi-6],0

	movzx edx,byte [vehbase+eax]

	lodsb		// n-vid
	and al,0x7f
	add esi,eax

	// translate cargo-IDs into actual sprite numbers
	lodsb
	mov ecx,eax

.nextcid:
	jecxz .defcid
	lodsb		// cargo type

.defcid:
	lodsw		// cargo ID
	test ah,ah
	jnz newcargoid.invalid

	mov ax,[cargoids+2*eax]
	mov [esi-2],ax	// store sprite number instead
	dec ecx
	jns .nextcid
	ret

setvehcargomap:
	movzx edx,byte [vehbase+eax]
	mov ebx,esi

	lodsb
	mov ecx,eax
	and ecx,0x7f

	mov [ebx-6],cl	// reset num-overrides

	xor al,0x80
	js .regularmap

	mov ebx,[lastnonoverride]
	mov bh,[ebx+1]


	// record wagon override
.nextwagon:
	lodsb

	cmp al,bh
	sete bl
	inc bl
	mov [wagonoverride+eax+edx],bl
	loop .nextwagon

	mov eax,[lastnonoverride]
	or byte [eax-3],0x80
	inc byte [eax-3]
	ret

.regularmap:
	mov [lastnonoverride],ebx
	lea eax,[edi-1]
	mov [ebx-5],ax
	xor eax,eax
	mov [ebx-3],al

	lea edx,[vehids+edx*8]
	mov ebp,[curspriteblock]

	// record offset to cargo-ID table for all the veh.IDs
.nextvid:
	lodsb
	mov [edx+8*eax],ebx
	mov [edx+8*eax+4],ebp
	loop .nextvid
	ret


	// *** action 4 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

applynewvehnames:
%if newtrains+0x48<>anyflagset
	%error "Somebody messed with newtrains! Leave it at 55!"
%endif
	movzx ebx,byte [vehbase+eax]
	lodsb

	btr eax,7		// general texts instead of vehicle names?
	sbb edx,edx		// 0 for vehnames, -1 for general texts

	mov ecx,[languageid]
	inc ecx
	shr al,cl
	jc .rightlanguage
	ret

.rightlanguage:
	lodsb
	mov ecx,eax		// num-veh

	test edx,edx
	js .generaltext

	lodsb
	add ebx,eax		// offset

	mov ah,0x80
	call gettexttableptr
	lea ebx,[eax+ebx*4]
	jmp short .gotit

.generaltext:
	push edi
	lodsw
	call gettextintableptr
	lea ebx,[eax+edi*4]
	pop edi

.gotit:

.nextveh:
	mov [ebx],esi
	add ebx,byte 4

.nextbyte:
	push ecx
	or ecx,byte -1
.nextnull:
	lodsb
	test al,al
	loopnz .nextnull
	pop ecx
	loop .nextveh
	ret


	// *** action 5 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

initnewgraphicsblock:
	lodsb
	add edi,eax	// skip this block

	mov ebx,[newgraphicsspritenums+ecx-4*4]
	test ebx,ebx
	jle .skip

	cmp dword [ebx],0
	jle .norequirednumber

	cmp eax,[ebx]
	jne near recordgrfid.wronggrfversion

.norequirednumber:
	mov [ebx],eax
.skip:
	ret

activatenewgraphics:
	mov ebx,[newgraphicsspritebases+ecx-4*4]
	test ebx,ebx
	jle newcargoid.invalid	// signed or zero = bad value

	lodsb
	xchg eax,ecx
	call insertactivespriteblock

	mov [ebx],ax
	ret
; endp newgraphics


	// *** action 6 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

applyparam:
	// here ecx=number*4

	lodsb			// eax=size
	lea ebx,[ecx+eax+3]	// now ebx=number*4+size+3
	shr ebx,2		// now ebx=required number of params
	cmp bl,[edx+spriteblock.numparam]
	ja .dont

	mov ebx,[edx+spriteblock.paramptr]
	add ebx,ecx		// now ebx points to param value

	mov ecx,eax
	lodsb
	push esi
	push edi
	mov esi,[edx+spriteblock.spritelist]
	mov edi,[esi+edi*4]
	add edi,eax
	mov esi,ebx
	rep movsb
	pop edi
	pop esi

	jmp short .next

.dont:
	lodsb		// need to skip offset

.next:
	lodsb
	mov ecx,eax
	shl ecx,2
	cmp al,0xff
	jnz applyparam
.done:
	ret


	// *** action 7 and 9 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

skipspriteif:
	test al,al
	jns .isparam

	mov edx,[externalvars+ecx-0x80*4]
	mov bh,1
	jmp short .gotvalue

.isparam:
	cmp al,[edx+spriteblock.numparam]
	jae .dont

	mov edx,[edx+spriteblock.paramptr]
	add edx,ecx		// now [edx]=param value
	mov bh,0

.gotvalue:
	lodsb

.gotsize:
	mov ecx,eax		// size
	lodsb

.gottest:
	sub al,2
	xchg eax,ebx		// ebx=condition type
	jb .bittest

	mov edx,[edx]

	mov bh,ah		// bh=1 if externalvar, 0 if param

	// a value test
	mov eax,[esi]
	add esi,ecx

	shl ecx,3
	neg ecx
	add ecx,32		// now ecx=32-8*size

	shl eax,cl
	shr eax,cl		// clear unset bits of eax

	test bh,bh
	jz .notgamevar

	shl edx,cl		// clear unset bits of edx too
	shr edx,cl
	mov bh,0

.notgamevar:
	// now eax = value
	lea ebx,[addr(.comptests)+ebx*4]
	cmp eax,edx
	jmp ebx

	// each case must be 4 bytes here (except last one)
.comptests:	// 2 = equal
	je .skipit
	jmp short .dont		// "jmp" isn't short by default... stupid...

		// 3 = not equal
	jne .skipit
	jmp short .dont

		// 4 = greater
	ja .skipit
	jmp short .dont

		// 5 = less
	jb .skipit
	jmp short .dont

		// 6 = GRFID is active
	mov bl,1
	jmp short .findgrfid

		// 7 = GRFID is inactive
	mov bl,0
	// jmp short .findgrfid

.findgrfid:
	mov bh,bl
	not bh

	test edx,edx
	jz .gotgrfid	// bh is such that bh!=bl

	mov bh,[edx+spriteblock.active]
	cmp eax,[edx+spriteblock.grfid]
	mov edx,[edx+spriteblock.next]
	jne .findgrfid

.gotgrfid:
	cmp bl,bh
	je .skipit

.dont:
	ret

.bittest:
	xor eax,eax
	lodsb
		// eax=bit number
		// ebx=bit test type-2
		// [edx]=value

	bt [edx],eax	// use memory bit test so we can use bitnumbers>31

	adc bl,1	// Have cases:
			// test	case	bl in  carry  bl out  skip?
			// 0	set	-2	  0	 -1	 no
			// 0	set	-2	  1	 0	 yes
			// 1	not set	-1	  0	 0	 yes
			// 1	not set	-1	  1	 1	 no
	jnz .dont

.skipit:
	xor eax,eax
	lodsb
	test al,al
	jnz .ok

	or edi,byte -1

.ok:
	add edi,eax
	ret


	// *** action 8 handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

grfidactivate:
	mov ah,1	// ah=1 to activate, ah=0 (by default) to record

recordgrfid:
	mov cl,ah
	lea ebx,[esi-1]
	mov [edx+spriteblock.action8],ebx

	cmp al,THISGRFVERSION
	jb .wronggrfversion

	cmp al,MAXGRFVERSION
	jna .isok

.wronggrfversion:
	mov ax,ourtext(wronggrfversion)
	call setspriteerror
.skip:
	or edi,byte -1
	ret

.isok:
	lodsd		// grf-ID

	mov [edx+spriteblock.grfid],eax

	test cl,cl
	jnz .activate
	ret

.activate:
	and byte [edx+spriteblock.active],~2		// mark as processed
	cmp [edx+spriteblock.active],cl
	jne .skip	// skip rest of file if not active
	ret


	// *** action A handler ***

	// Four extra bytes in front of this sprite data are used this way:
	// - unused

replacettdsprite:
	mov ebx,eax		// num-sets

.nextset:
	lodsb
	movzx ecx,al		// num-sprites

	lodsw			// first-sprite

	lea ebp,[eax+ecx]
	cmp ebp,totalsprites
	jae newcargoid.invalid

	mov ebp,edi
.replnext:
	pusha
	mov edi,[edx+spriteblock.spritelist]
	mov esi,[edi+ebp*4]
	mov edi,[esi-4]		// sprite size
	xchg eax,edi
	call overridesprite
	popa
	inc edi
	inc eax
	inc ebp
	loop .replnext

	dec ebx
	jnz .nextset
	ret

replacettdspriteskip:
	mov ecx,eax
.nextset:
	lodsb
	add edi,eax
	lodsb
	loop .nextset
	ret


// insert active action 1 or action 5 sprites into TTD's sprite number space
//
// in:	ecx=number of sprites
//	edx=>current sprite block
//	edi=number of first sprite following action 1 or 5 in this block
// out:	eax=real number of first sprite
//	edi adjusted properly
//
// sets spriteerror and returns edi=-1 on error
//
insertactivespriteblock:
	mov eax,[numactsprites]
	add eax,ecx
	cmp ah,0x40
	jb .nottoomany

	mov ax,ourtext(toomanyspritestotal)
	call setspriteerror
	or edi,byte -1
	ret

.nottoomany:
	xchg eax,[numactsprites]

	push eax

.replnext:
	pusha
	mov esi,[edx+spriteblock.spritelist]
	mov esi,[esi+edi*4]
	mov edi,[esi-4]		// sprite size
	xchg eax,edi
	call overridesprite
	popa
	inc edi
	inc eax
	loop .replnext

	pop eax
	ret



// load sprite header
// TTD normally respects the "immutable" flag, which fixes the sprite cache
// position of a sprite. However, everytime a new game is started, it resets
// all sprites, even immutable ones.  We need to preserve those.
// Note: TTD doesn't ever set the immutable flag itself.
//
// Don't load header if returning with carry set.
//
// in:	ax=sprite X size from file
//	esi=sprite number
// out:	set spritexsize[esi*2] to ax if appropriate
// safe:-
loadspriteheader:
	push edx
	imul edx,[newspritenum],19
	add edx,[newspritedata]
	cmp byte [edx+esi],1		// is the immutable flag set?
	cmc
	jc .done

	imul edx,[newspritenum],6
	add edx,[newspritedata]
	mov [edx+esi*2],ax

.done:
	pop edx
	ret


// overridesprite: replace a TTD sprite with a new sprite
//
// in:	eax=sprite data size including header
//	edi=sprite number
//	esi=>sprite data
// out: esi=>after sprite data
// uses:all
//
// overrideembeddedsprite: same, but size and number are stored at esi:
//	W spritenum
//	W spritesize (as stored in TTD's cache)
//	B[spritesize] spritedata

overrideembeddedsprite:
	xor eax,eax
	lodsw
	mov edi,eax
	lodsw

overridesprite:
	mov ebx,[newspritedata]
	mov ebp,[newspritenum]

	cmp dword [ebx+edi*4],0	// is it currently in the cache?
	je .notincache

	imul ecx,ebp,19
	add ecx,edi
	cmp byte [ebx+ecx],0	// was it immutable (i.e. one of ours)?
	jne .notincache

#if !WINTTDX
		// load the right selector in case we have no new graphics
	push es
	mov es,[spritecacheselector]
#endif
	push esi
	mov ecx,edi
	call [removespritefromcache]
	pop esi
#if !WINTTDX
	pop es
#endif

.notincache:
	mov [ebx+edi*4],esi	// set cache offset

	call setspriteinfo

	imul ebp,7
	add ebx,ebp
	mov byte [ebx+edi],1	// set immutable

	add esi,ecx
	ret

// reload sprite info of sprite that had override
//
// in:	edi=sprite number
//
reloadspriteheaders:
	mov ebx,[newspritedata]
	mov ebp,[newspritenum]

	imul ecx,ebp,18
	add ecx,ebx
	movzx edx,byte [ecx+edi]

	mov esi,[curfileofsptr]
	sub esi,8
	push esi

	mov ax,[esi-8+edx*2]
	mov [tempspritefilehandle], ax

	mov byte [curdecoderuntype],0
	mov word [curfileblocksize],0

	push ebx
	imul ecx,ebp,14
	add ecx,ebx
	mov edx,[ecx+edi*4]
	sub edx,2
	mov ecx,edx
	shr ecx,10h
	mov ax,4200h
	mov bx,[tempspritefilehandle]
	CALLINT21

	call dword [readwordfn]
	pop ebx
	lea ecx,[ebx+ebp*4]
	mov [ecx+edi*2],ax
	push eax

	mov ax,8
	call dword [readspriteinfofn]

	pop eax
	pop esi

setspriteinfo:
	lea ebx,[ebx+ebp*4]

	mov [ebx+edi*2],ax	// set data size
	lea ebx,[ebx+ebp*2]
	lea ecx,[eax-8]		// data size minus header

	xor eax,eax
	lodsb			// skip sprite type (compression code)
	lodsb
	mov edx,eax
	lodsw
	mov [ebx+edi*2],ax	// set x size
	lea ebx,[ebx+ebp*2]

	mov [ebx+edi*2],dx	// set y size
	lea ebx,[ebx+ebp*2]

	lodsw
	mov [ebx+edi*2],ax	// set x offset
	lea ebx,[ebx+ebp*2]

	lodsw
	mov [ebx+edi*2],ax	// set y offset
	ret


	// make a list of grf IDs, or add new IDs to the existing list
	//
	// make sure all sprite blocks have .active set correctly; it will
	// be copied into the list for any new IDs
makegrfidlist:
	pusha

	xor ebx,ebx
	lea ecx,[ebx+5]
	xchg ebx,[grfidlistnum]
	call makegrfidlistsize		// make sure the pointer is valid
	jc .done

	mov esi,[spriteblockptr]

.nextblock:
	mov eax,[esi+spriteblock.grfid]
	test eax,eax
	jz .gotit	// skip null IDs

	cmp eax,byte -1
	je .gotit	// and GRFIDs of -1

	mov edi,[grfidlist]

	mov ecx,ebx
	jecxz .notfound

.next:
	scasd
	je .gotit

	scasb	// skip "activated" byte
	loop .next

	// ID was not found; add it to the list
.notfound:
	lea ecx,[ebx*5+5]
	call makegrfidlistsize
	jc .done

	lea edi,[ebx*5]		// need to calculate again in case makegrfidlistsize
	add edi,[grfidlist]	// moved it because it was too small
	stosd
	mov al,[esi+spriteblock.active]
	stosb
	inc ebx

.gotit:
	mov esi,[esi+spriteblock.next]
	test esi,esi
	jnz .nextblock

	mov [grfidlistnum],ebx

.done:
	popa
	ret

uvard grfidlistsize	// size in bytes of memory allocated to the list
uvard grfidlist		// pointer to the list
uvard grfidlistnum	// number of entries in the list (5 bytes each)
uvarb activatedefault	// whether to activate by default or not


	// make sure the grfidlist has a minimum size (given by ecx)
makegrfidlistsize:
	cmp [grfidlistsize],ecx
	jb .makelarger
	ret

.makelarger:
	pusha
	add ecx,1023
	and ecx,-1024	// allocate in chunks of 1024 bytes

	push ecx
	call spritemalloc
	pop edi
	jc .done	// out of memory

	mov esi,edi

	xchg ecx,[grfidlistsize]
	xchg esi,[grfidlist]

	rep movsb

.done:
	popa
	ret


	// go through all sprite blocks and choose which
	// ones to activate from the list in the grf ID list
	// also update the active byte in the list
setactivegrfs:
	pusha

	xor ebx,ebx
	xchg ebx,[grfidlistnum]

	mov esi,[spriteblockptr]

.nextblock:
	mov eax,[esi+spriteblock.grfid]
	cmp byte [esi+spriteblock.active],0x80
	je .skipblock

	test eax,eax
	jz .notfound

	cmp eax,byte -1
	je .alwaysactive

	mov edi,[grfidlist]

	mov ecx,ebx
	jecxz .notfound

.next:
	scasd
	je .gotit

	scasb	// skip "activated" byte
	loop .next

.notfound:
	// ID was not found; use default
	mov al,[activatedefault]
	jmp short .gotactive

.alwaysactive:
	// ID is a special ID that is always active
	mov al,1
	jmp short .gotactive

.gotit:
	mov al,[edi]

.gotactive:
	add al,2
	mov [esi+spriteblock.active],al

.skipblock:
	mov esi,[esi+spriteblock.next]
	test esi,esi
	jnz .nextblock
	popa

	ret



// sizes of each entry in the vehicle specific data tables
// format: total, sizes...
// total is the sum of all sizes in the list, anything beyond that
// is handled by the vehicle specific subroutine.
// this should be probably moved to vars.ah, eventually

%define d_B 1		// a byte
%define d_W 2,0		// a word
%define d_D 4,0,0,0	// a dword
%define d_P 0x84,0,0,0	// a pointer, relative to the data segment (for WINTTDX)

%macro defvehdata 1-*.nolist	// params: name,arraysizes...
	var %1
	%assign %%totalsize 0
	%rotate 1
	%rep %0-1
		%ifidn {%1},{B}
			%assign %%totalsize %%totalsize+1
		%elifidn {%1},{W}
			%assign %%totalsize %%totalsize+2
		%elifidn {%1},{D}
			%assign %%totalsize %%totalsize+4
		%elifidn {%1},{P}
			%assign %%totalsize %%totalsize+4
		%else
			%define arrsize %1
			%error "Invalid array size arrsize"
		%endif
		%rotate 1
	%endrep
	db %%totalsize
	%rotate 1
	%rep %0-1
		db d_%1
		%rotate 1
	%endrep
	%1_totalsize equ %%totalsize
%endmacro

defvehdata gendata, W,B,B,B,B,B
defvehdata spectraindata, B,W,W,B,P,B,B,B,B,B,B,B
defvehdata specrvdata, B,B,P,B,B,B,B,B
defvehdata specshipdata, B,B,B,B,B,W,B,B
defvehdata specplanedata, B,B,B,B,B,B,B,W,B,B

%undef defvehdata
%undef d_B
%undef d_W
%undef d_D
%undef d_P

uvarb spriteand
uvard spritebase

uvard numactivesprites
uvard lastnonoverride
uvarb grfstage

	// pointers to the actions specified in the first byte of
	// the pseudo sprite data

	align 4
var spriteinitializeaction
	dd 0				// 0: new vehicle data
	dd addr(newspriteblock)		// 1: sprite block
	dd addr(newcargoid)		// 2: cargo ID
	dd addr(initializevehcargomap)	// 3: veh ID->cargo ID mapping
	dd 0				// 4: new veh name
	dd addr(initnewgraphicsblock)	// 5: non-veh sprite block
	dd addr(applyparam)		// 6: apply parameter
	dd 0				// 7: skip sprites if condition true
	dd addr(recordgrfid)		// 8: grf ID
	dd addr(skipspriteif)		// 9: skip sprites even during init
	dd addr(replacettdspriteskip)	// A: replace TTD's sprites

numspriteactions equ (addr($)-spriteinitializeaction)/4

var spriteactivateaction
	dd addr(processnewinfo)		// 0: new vehicle data
	dd addr(activatevehspriteblock)	// 1: sprite block
	dd addr(activatecargoid)	// 2: cargo ID
	dd addr(setvehcargomap)		// 3: veh ID->cargo ID mapping
	dd addr(applynewvehnames)	// 4: new veh name
	dd addr(activatenewgraphics)	// 5: non-veh sprite block
	dd 0				// 6: apply parameter
	dd addr(skipspriteif)		// 7: skip sprites if condition true
	dd addr(grfidactivate)		// 8: grf ID
	dd addr(skipspriteif)		// 9: skip sprites even during init
	dd addr(replacettdsprite)	// A: replace TTD's sprites

%ifndef PREPROCESSONLY
%if numspriteactions <> (addr($)-spriteactivateaction)/4
	%error "Inconsistent number of sprite actions"
%endif
%endif


	// the following variables need to be close in memory
var vehbase, db TRAINBASE,ROADVEHBASE,SHIPBASE,AIRCRAFTBASE
var vehbnum, db NTRAINTYPES,NROADVEHTYPES,NSHIPTYPES,NAIRCRAFTTYPES

	// action 0 properties defined so far
var maxprop, db 0x21,0x16,0x11,0x12

	// for action 0, where are the regular vehicle specific properies listed
var specificpropertylist, dd spectraindata,specrvdata,specshipdata,specplanedata

	// for action 0, where the data for each vehicle class starts
	// (set by patching functions)
var specificpropertybase, times 4 dd -1

	// and how much these are offset from the sprite bases
var specificpropertyofs, db -10,-6,0,0

	// for action 0, which procedure to call for special properties
var specialpropertyfn, dd addr(proctraindata),addr(procrvdata),addr(procshipdata),-1

	// pointer to where the vehicle ID has the cargo IDs specified
	// followed by a pointer to the corresponding sprite block
uvard vehids, 256*2

	// record whether waggons have override (must immediately follow the
	// vehids declaration)
uvard wagonoverride, 256/4	// "1" for wagons that have override

	// and the list of cargo IDs
uvarw cargoids, 256

	// for action 5, where to store the first sprite number
var newgraphicsspritebases, dd presignalspritebase,catenaryspritebase,extfoundationspritebase,morestationspritebase,-1
var newgraphicsspritenums, dd numsiggraphics,numelrailsprites,extfoundationspritenum,nummorestationgraphics,-1

// cargo types available in each scenario
var scenariocargo, dd 0x7ff,0x1cff,0x1f4ed,0x7fe0005

// cargo types for refitting, for all climates
var cargotypes
	db  0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,-1	// Temperate
	db  0, 1, 2, 3, 4, 5, 6, 7,-1,11,10,12	// Arctic
	db  0,16, 2, 3,13, 5, 6, 7,14,15,10,12	// Tropic
	db  0,17, 2,18,19,20,21,22,23,24,25,26	// Toyland

// and the reverse of the above list
var cargoid, db 0,1,2,3,4,5,6,7,8,9,10,9,11,4,8,9,1,1,3,4,5,6,7,8,9,10,11

// replacement table for TTD sprites
// contents for each original sprite:
//	   0..131A = new TTD sprite
//	2000..7FFF = new TTDPatch sprite
//	8000+x	   = call runindexbase+4*x with eax=old sprite; should return
//			ebx=new sprite (possible values 0..131A or 2000..7FFF)
uvarw newttdsprites, totalsprites

uvard setcharwidthtablefn,1,s	// TTD function to set character width table based on the sprite data
				// has to be called again if characters are changed via the new .GRF files
