// New loading and saving functions to support more vehicles

// Known (defined) extra chunks.
// The first table defines chunk IDs.
var knownextrachunkids
// Optional chunks
	dw 0		// TTD Vehicle data
	dw 1		// TTDPatch Vehicle data
	dw 2		// GRF IDs that we know about
	dw 3		// TTDPatch version and configuration
// Mandatory chunks (needed for some features to work)
	dw 0x8000	// Town array extension
knownextrachunknum equ (addr($)-knownextrachunkids)/2

// The load functions for each chunk
var knownextrachunkloadfns
	dd addr(loadspecvehdata)
	dd addr(loadttdvehdata)
	dd addr(loadgrfidlist)
	dd addr(skipchunkonload)
	dd addr(loadtown2array)
%ifndef PREPROCESSONLY
%if knownextrachunknum <> (addr($)-knownextrachunkloadfns)/4
	%error "Inconsistent number of chunk functions"
%endif
%endif

// The save functions for each chunk
var knownextrachunksavefns
	dd addr(savespecvehdata)
	dd addr(savettdvehdata)
	dd addr(savegrfidlist)
	dd addr(savepatchconfig)
	dd addr(savetown2array)
%ifndef PREPROCESSONLY
%if knownextrachunknum <> (addr($)-knownextrachunksavefns)/4
	%error "Inconsistent number of chunk functions"
%endif
%endif

// The query functions for each chunk (to determine if the chunk is to be saved/loaded)
var knownextrachunkqueryfns
	dd addr(canhavespecvehdata)
	dd addr(canhavettdvehdata)
	dd addr(canhavegrfidlist)
	dd addr(canhavepatchconfig)
	dd addr(canhavetown2array)
%ifndef PREPROCESSONLY
%if knownextrachunknum <> (addr($)-knownextrachunkqueryfns)/4
	%error "Inconsistent number of chunk functions"
%endif
%endif



// Temporary variables used in load/save code
uvarb chunkstosave,(knownextrachunknum+8)/8	// bit map of chunks to be saved (in reverse order)
				// +8 not +7 because bits are indexed by loop counter...
uvarb loadstage		// used to retry the loading after checking veh. multiplier if lowmemory
uvarb loadproblem
uvarb loadreduced	// nonzero: had to reduce vehicle array from this value
uvarw loadremovedvehs	// had to remove this many vehicles
uvarw loadremovedcons	// ...in this many consists (these 2 vars are also accessed as a single DWORD)
uvarw loadremovedsfxs	// ... and this many pseudo-/special vehicles

%assign LOADPROBLEM_UNKNOWNCHUNK	0x1
%assign LOADPROBLEM_BADCHUNK		0x2
%assign LOADPROBLEM_RCHUNKNOTLOADED	0x4	// required (mandatory) chunk present but not loaded

// Additional flags to indicate whether some specific data have been loaded
//%assign LOADED_X1_TTDVEHDATA		0x?	// ** currently unused **
%assign LOADED_X1_TOWN2			0x1

uvarb extrachunksloaded1		// a combination of LOADED_X1_*


struc extrachunkhdr
	.id:	resw 1
	.length:resd 1
endstruc_32

var extrachunkheader
	istruc extrachunkhdr
		dw -1
		dd 0
	iend


%macro CALLLOADSAVECHUNK 0
	call ebp		// AFAICT [loadchunkfn] and [savechunkfn] preserve everything except AL,ECX,ESI
%endmacro

// Load or save the main data. In: EBP->load or save function, ESI->startdata, BL = 0=load, 1=save
loadsavemaindata:
	// first chunk: up to oldveharray
	// here's where ESI pointing to datastart comes in handy
	mov ecx,oldveharray-datastart
	push ds
	pop es
	CALLLOADSAVECHUNK

	// second chunk: the vehicle array
	mov esi,[veharrayptr]
	test bl,1
	jnz .saving

	// we're loading; make sure we load exactly as much as we need
	movzx eax,byte [landscape3+ttdpatchdata.vehfactor]
	cmp al,2
	adc al,0			// 0->1, 1->2, other values left as they are
	mov [loadreduced],al		// will be checked later and zeroed if OK
	imul ecx,eax,oldveharraysize
	jmp short .dosecondchunk

.saving:
	// we're saving; here it's simple, just save everything
	mov ecx,[veharrayendptr]
	sub ecx,esi

.dosecondchunk:
	CALLLOADSAVECHUNK

	// third chunk: the rest
	mov esi,oldveharray+oldveharraysize
	mov ecx,datasaveend-(oldveharray+oldveharraysize)
	CALLLOADSAVECHUNK

	// fourth chunk: the L4 array
#if WINTTDX
	mov esi,landscape4base
#else
	xor esi,esi
	push fs
	pop es
#endif
	mov ecx,0x10000
	CALLLOADSAVECHUNK

	// fifth chunk: the L5 array
#if WINTTDX
	mov esi,landscape5base
#else
	xor esi,esi
	push gs
	pop es
#endif
	mov ecx,0x10000
	CALLLOADSAVECHUNK

#if !WINTTDX
	push ds
	pop es
#endif
	ret


// New save function
// in:	ESI -> datastart
newsaveproc:
	mov dword [landscape3+ttdpatchdata.magic],L3MAGIC	// indicate new TTDPatch extra data format

	// record the vehicle array multiplier
	mov al,[vehicledatafactor]
	cmp al,1
	jne .isnotone
	mov al,0			// 1,0->0  2 etc. ok

.isnotone:
	mov [landscape3+ttdpatchdata.vehfactor],al

	// record the number of extra chunks
	mov ebx,chunkstosave
	mov edx,knownextrachunkqueryfns
	xor ecx,ecx
	mov cl,knownextrachunknum	// if knownextrachunknum exceeds 255 in some far future, NASM will complain
	xor eax,eax

.extrachunkpreploop:
	btr [ebx],ecx
	pusha
	stc
	call [edx]
	popa
	jnc .prepnextchunk
	bts [ebx],ecx
	inc eax

.prepnextchunk:
	add edx,byte 4
	loop .extrachunkpreploop

	mov [landscape3+ttdpatchdata.extrachunks],ax

	// save the main data
	mov edx,addr(adjaivehicleptrssave)
	call adjaivehicleptrs

	mov ebp,[savechunkfn]
	mov bl,1
	call loadsavemaindata

	mov edx,addr(adjaivehicleptrsload)
	call adjaivehicleptrs

	cmp word [landscape3+ttdpatchdata.extrachunks],byte 0
	jz .noextrachunks

	// save all the extra chunks
	mov ebx,extrachunkheader
	mov esi,knownextrachunkids
	mov edx,knownextrachunksavefns
	xor ecx,ecx
	mov cl,knownextrachunknum	// if knownextrachunknum exceeds 255 in some far future, NASM will complain

.extrachunkloop:
	lodsw
	bt [chunkstosave],ecx
	jnc .nextchunk
	mov [ebx+extrachunkhdr.id],ax
	pusha
	call [edx]
	popa

.nextchunk:
	add edx,byte 4
	loop .extrachunkloop

.noextrachunks:
	// we're through
	ret
; endp newsaveproc


// New load function
// in:	ESI -> datastart
// out: CF=1 - failed
//	CF=0 and ZF=1 - rewind, otherwise finished
newloadproc:
	push ds
	pop es
	mov ebp,[loadchunkfn]

	// if lowmemory is on, check if vehicle array will fit
	// abort with "Game Load Failed" if no, rewind the file and proceed with loading if yes
	mov al,0
	xchg al,[loadstage]
	or al,al
	jnz .proceedwithload		// already checked the veharray size

	testflags lowmemory
	jnc .proceedwithload		// in the normal memory mode the array *should* always suffice

	mov esi,loadreduced
	mov ecx,landscape3+ttdpatchdata.vehfactor+1-datastart

.vehfactorloadloop:
	pusha
	xor ecx,ecx
	inc ecx				// load 1 byte at a time
	call ebp
	popa
	loop .vehfactorloadloop

	mov al,[esi]
	cmp al,2
	adc al,0			// 0->1, 1->2, other values left as they are
	cmp [vehicledatafactor],al	// CF=1 if too large to fit
	jc .rewindorabort
	mov byte [loadstage],1
	xor al,al

.rewindorabort:
	ret

.proceedwithload:
	and byte [loadproblem],0

	// start by resetting all the vehicle type data
	push ebp
	push esi

	call inforeset
	and byte [extrachunksloaded1],0	// indicate that extra data are no longer valid
	and dword [grfidlistnum],0	// clear list of GRF IDs
	mov byte [activatedefault],1	// activate all by default if we have no grf id list block

	call setvehiclearraysize	// probably unnecessary but just in case
	call cleardata

	pop esi
	pop ebp

	// load the main data
	mov bl,0
	call loadsavemaindata

	mov edx,addr(adjaivehicleptrsload)
	call adjaivehicleptrs

	movzx ecx,word [landscape3+ttdpatchdata.extrachunks]

	// convert the old-style TTDPatch extra data in L3 if necessary
	cmp dword [landscape3+ttdpatchdata.magic],L3MAGIC
	je .ttdpatchdataOK

	mov esi,landscape3
	setbase edi,landscape3+ttdpatchdata.start
	mov eax,[esi+ttdpatchdataold.chtused]
	mov [BASE landscape3+ttdpatchdata.chtused],eax
	mov eax,[esi+ttdpatchdataold.orgpl1]
	mov [BASE landscape3+ttdpatchdata.orgpl1],ax	// also copies .orgpl2
	mov eax,[esi+ttdpatchdataold.yearsadd]
	mov [BASE landscape3+ttdpatchdata.yearsadd],ax
	mov eax,[esi+ttdpatchdataold.daysadd]
	mov [BASE landscape3+ttdpatchdata.daysadd],eax
	mov al,[esi+ttdpatchdataold.realcurrency]
	mov [BASE landscape3+ttdpatchdata.realcurrency],al
	setbase none

	movzx ecx,word [esi+ttdpatchdataold.extrachunks]

.ttdpatchdataOK:
	// now, are there any extra chunks to load or skip?
	jecxz .noextrachunks

	// there are extra chunks, load or skip them as appropriate
	mov ebx,extrachunkheader

.extrachunkloop:
	pusha

	// load a chunk's header
	mov esi,ebx
	xor ecx,ecx
	mov cl,extrachunkhdr_size
	call ebp

	// try to identify the chunk
	mov dx,[ebx+extrachunkhdr.id]
	mov eax,[ebx+extrachunkhdr.length]
	xor ecx,ecx

.findchunk:
	cmp word [knownextrachunkids+ecx*2],dx
	je .found
	inc ecx
	cmp ecx,0+knownextrachunknum	// leading 0+ makes the macro in opimm8.mac work
	jb .findchunk

	// not found, skip the chunk
	or byte [loadproblem],LOADPROBLEM_UNKNOWNCHUNK

.skipchunk:
	call skipchunkonload
	jmp short .nextchunk

.found:
	pusha
	clc
	call dword [knownextrachunkqueryfns+ecx*4]
	popa
	jnc .skipchunk

	call dword [knownextrachunkloadfns+ecx*4]

.nextchunk:
	popa
	loop .extrachunkloop

.noextrachunks:
	// find engines in consists
	mov esi,loadreduced
	movzx ebx,byte [esi]		// now EBX=loaded array's multiplier
	mov edi,veharrayendptr
	push dword [edi]
	mov eax,[veharrayptr]
	imul ebp,ebx,oldveharraysize
	add ebp,eax			// now EBP->loaded array's end
	mov [edi],ebp			// temporarily set veharrayendptr to this value

	call findengines

	// check the loaded veharray's multiplier
	// EAX->vehicle array, EBX=loaded array's multiplier, EBP->loaded array's end,
	// ESI->loadreduced, EDI->veharrayendptr
	movzx ecx,byte [vehicledatafactor]
	cmp bl,cl
	ja .reduceveharray
	mov byte [esi],0
#if WINTTDX
	jmp .veharrayok
#else
	jmp short .veharrayok
#endif

.reduceveharray:
	// reduce the vehicle array to vehicledatafactor*850 entries
	// EAX->vehicle array, EBX=loaded array's multiplier, ECX=vehicledatafactor,
	// ESI->loadreduced, EDI->veharrayendptr, EBP->loaded array's end
#if WINTTDX
	xor edx,edx
	inc edx				// EDX=1, add datastart to pointers
	call adjustscheduleptrs
#endif

	mov edx,loadremovedvehs
	and dword [edx],byte 0
	and word [byte edx+loadremovedsfxs-loadremovedvehs], byte 0

	imul ecx,ecx,oldvehicles
	mov esi,dayprocnextveh
	cmp [esi],cx
	jbe short .procvehOK
	mov [esi],cx

.procvehOK:
	shl ecx,vehicleshift
	add ecx,eax			// now ECX->start of the discarded area of vehicle array

.consistloop:
	cmp byte [ecx+veh.class],0
	je short .nextconsist

	movzx esi,word [ecx+veh.engineidx]
	shl esi,vehicleshift
	add esi,eax

	cmp byte [esi+veh.class],0x14
	jae short .delspec
	inc word [byte edx+loadremovedcons-loadremovedvehs]

.delloop:
	inc word [edx]
	jmp short .delschedule

.delspec:
	inc word [byte edx+loadremovedsfxs-loadremovedvehs]

.delschedule:
	cmp dword [esi+veh.scheduleptr],byte -1
	jz short .delveh
	pusha
	mov edx,esi
	call [delvehschedule]
	popa

.delveh:
	call dword [delveharrayentry]		// preserves all registers
	movzx esi,word [esi+veh.nextunitidx]
	cmp si,byte -1
	je short .nextconsist

	shl esi,vehicleshift
	add esi,eax
	cmp byte [esi+veh.class],0x14
	jae short .delspec
	jmp short .delloop

.nextconsist:
	sub ecx,byte -vehiclesize
	cmp ecx,ebp
	jb short .consistloop

#if WINTTDX
	or edx,byte -1				// EDX=-1, subtract datastart back from pointers
	call adjustscheduleptrs
#endif

.veharrayok:
	// still EAX->vehicle array, EBX=loaded array's multiplier,
	// EDI->veharrayendptr, EBP->loaded array's end
	pop dword [edi]			// restore veharrayendptr

	// check validity of some variables
	mov eax,[landscape3+ttdpatchdata.orgpl1]
	cmp al,ah
	jne short .validcompanies

	mov eax,[human1]
	mov [landscape3+ttdpatchdata.orgpl1],ax

.validcompanies:
	// now check extra arrays (which typically go to mandatory chunks)
	// if any is needed but has not been loaded, initialize it

	// check town array extension
	mov eax,[townarray2ofst]
	test eax,eax
	jz .town2array_ok
	test byte [extrachunksloaded1],LOADED_X1_TOWN2
	jnz .town2array_ok

	// town array extension is used but was not loaded
	// initialize using the standard data
	lea edi,[eax+townarray]
	mov esi,edi			// save for later
	mov ecx,numtowns*town2_size/4
	xor eax,eax
	rep stosd			// clear the array first

	call recalctownpopulations

	// convert the old-style bribe fail data
	mov edi,townarray
	xor ecx,ecx
	mov cl,numtowns

.copybribedata:
	movzx edx,byte [edi+0x2f]
	bsf edx,edx			// check if a company is unwanted (old-style data can have only one)
	jz .nextbribedata

	// check for how long a company, if any, will be unwanted yet
	mov ax,[edi+0x30]
	sub ax,[currentdate]
	jbe .nextbribedata		// already expired

	// convert to months
	mov bl,30
	cmp ah,bl
	jae .bribemonthsoverflow
	div bl
.bribemonthsoverflow:

	inc eax
	mov [esi+town2.companiesunwanted+edx],al

.nextbribedata:
	and dword [edi+0x2e],0xff	// clear the old-style data
	add esi,0+town_size
	add edi,0+town_size
	loop .copybribedata

.town2array_ok:

	// looks like it's all. Whew!

	// now initialize the newgrfs.  this needs to run after loading
	// a game because the grfs may need to know some savegame data
	// like the climate

	call infoapply

	// finally make sure all vehicles have correct new sprites
	// in case newgrf.txt has changed

	call resetnewsprites

	or al,1				// indicate success (CF=ZF=0)
	ret
; endp newloadproc


// Save a chunk's header.  Assume [extrachunkheader.id] already contains the ID, and EAX=length
// also assumes ES=DS, EBP->save function
// preserves all registers
savechunkheader:
	pusha
	mov esi,extrachunkheader
	mov [esi+extrachunkhdr.length],eax
	xor ecx,ecx
	mov cl,extrachunkhdr_size
	call ebp
	popa
	ret


// Bad chunk (see below for registers)
badchunk:
	or byte [loadproblem],LOADPROBLEM_BADCHUNK
	// fallthrough

// Unknown or bad chunk, skip
// in:	EAX=length, EBP->load function, ES=DS
// uses:EAX,ECX,ESI
// note: there's some code that assumes EBP does not have to be valid if EAX=0
skipchunkonload:
	mov esi,extrachunkheader		// used here as a bit bucket
	test byte [esi+extrachunkhdr.id+1],0x80	// is it a non-optional chunk?
	jz .notrequired
	or byte [loadproblem],LOADPROBLEM_RCHUNKNOTLOADED

.notrequired:
	xor ecx,ecx
	inc ecx				// load byte by byte (the load function does same)
	inc eax
	jmp short .startloop		// EAX may be 0 on entry!

.skiploop:
	pusha
	call ebp
	popa

.startloop:
	dec eax
	jnz .skiploop
	ret


// Load, save and query functions for the extra chunks.
// All these functions are called with ES=DS.
// Load and save functions are called with:
//	EBX->extrachunkheader (for convenience)
// Load functions are additionally called with:
//	EBP=[loadchunkfn] (i.e. points at TTD's chunk load function)
//	EAX=number of bytes to read (=extrachunkheader.length)
// Save functions are additionally called with:
//	EBP=[savechunkfn] (i.e. points at TTD's chunk save function)
// Query functions are additionally called with:
//	CF=0 if loading, 1 if saving
// Load and save functions return no value.
// Query functions return with CF set (can save/load this chunk) or clear (do not save/load this chunk).
//
// Save functions are supposed to start with determining the size the chunk will have and calling savechunkheader.
// They *MUST* save at least a valid header, no matter what. The only way they're allowed to fail is by setting
// EAX to zero and jumping through savechunkheader (or an equivalent action), but it's better to fail
// in the corresponding query function.
//
// Load functions are called after a header has been read; if the chunk itself cannot or should not be read,
// they should call (or jump to) badchunk or skipchunkonload. In any case, the correct number of bytes must be read.
//
// All registers except the stack and segment registers are safe and need not be preserved.


// All query functions for chunks that depend only on saveoptdata being ON go here
canhavespecvehdata:
canhavettdvehdata:
canhavegrfidlist:
canhavepatchconfig:
	testflags saveoptdata
	ret



specvehdatalength equ \
 totalvehtypes*vehtypeinfo_size + \
 NTRAINTYPES*spectraindata_totalsize + \
 NROADVEHTYPES*specrvdata_totalsize + \
 NSHIPTYPES*specshipdata_totalsize + \
 NAIRCRAFTTYPES*specplanedata_totalsize

loadspecvehdata:
	cmp eax,specvehdatalength
	jne badchunk
	jmp short loadsavespecvehdata

savespecvehdata:
	mov eax,specvehdatalength
	call savechunkheader

	mov esi,[vehtypedataconvbackupptr]
	test esi,esi
	jnz short loadsavespecvehdata.usevehdatabackup

loadsavespecvehdata:
	mov esi,[vehtypedataptr]

.usevehdatabackup:
	mov ecx,totalvehtypes*vehtypeinfo_size
	call ebp

#if WINTTDX
	mov edx,datastart
	neg edx
	call adjspecvehdataptrs
#endif
	xor ebx,ebx

.saveloop:
	call getspecificpropertyarea		// sets ESI,ECX
	call ebp
	inc ebx
	cmp ebx,4
	jb .saveloop

.done:
#if WINTTDX
	mov edx,datastart
	call adjspecvehdataptrs
#endif
//*	or byte [extrachunksloaded1],LOADED_X1_TTDVEHDATA	// meaningless when saving
	ret

#if WINTTDX
adjspecvehdataptrs:
	xor ecx,ecx
	mov esi,[specificpropertybase]
	add esi,NTRAINTYPES*6
	mov cl,NTRAINTYPES
.loop1:
	cmp dword [esi],byte 0
	je .next1
	add [esi],edx
.next1:
	add esi,byte 4
	loop .loop1

	mov esi,[specificpropertybase+4]
	add esi,NROADVEHTYPES*2
	mov cl,NROADVEHTYPES
.loop2:
	add [esi],edx
	add esi,byte 4
	loop .loop2
	ret
#endif


// load and save TTDPatch vehicle data (from newvehdata struct)
loadttdvehdata:
	// this is probably superfluous, but it won't hurt either
	// the intent is mainly to make sure that if the data is from
	// an older version that saved fewer bytes, that the remainder
	// is initialized properly
	pusha
	call initttdpatchdata
	popa

	xor ebx,ebx
	mov ecx,newvehdatastruc_size
	sub ecx,eax
	jae .nottoomuch

	add eax,ecx
	sub ebx,ecx

.nottoomuch:
	jmp short loadsavettdvehdata

savettdvehdata:
	mov eax,newvehdatastruc_size
	call savechunkheader
	xor ebx,ebx

loadsavettdvehdata:
	mov esi,newvehdata
	xchg ecx,eax
	call ebp

	// we may have to get rid of surplus data on load
	// so that the following chunks (if any) will load properly
	xchg eax,ebx
	jmp skipchunkonload		// for now we don't indicate this condition; perhaps we should
					// also, this relies on skipchunkonload doing nothing when EAX=0...


// load the list of GRF IDs we know about
loadgrfidlist:
	mov ecx,eax
	jecxz .nothing

	xor edx,edx
	lea ebx,[edx+5]
	div ebx		// find out how many entries
	mov [grfidlistnum],eax

	call makegrfidlistsize

	mov esi,[grfidlist]
	call ebp

	// add grf IDs that we have now that weren't in the list before
	call makegrfidlist

	// with a GRF ID list loaded, only activate those .grfs that we know
	// unless loadallgraphics is on
	testflags loadallgraphics
	setc [activatedefault]

.nothing:
	ret


// save the list of GRF IDs we know about
savegrfidlist:
	call makegrfidlist
	imul eax,[grfidlistnum],5
	call savechunkheader
	mov esi,[grfidlist]
	mov ecx,eax
	jecxz .nothing
	call ebp
.nothing:
	ret


// save the current configuration and variables
savepatchconfig:
	mov eax,endflagdata - startflagdata + 4
	call savechunkheader

	mov esi,ttdpatchvercode
	mov ecx,4
	call ebp

	mov esi,startflagdata
	mov ecx,endflagdata - startflagdata
	call ebp
	ret


// Query, load and save functions for the town array 2
canhavetown2array:
	mov eax,[townarray2ofst]
	neg eax				// set CF if EAX is nonzero
	ret


loadtown2array:
	cmp eax,numtowns*town2_size
	jne badchunk
	jmp short loadsavetown2array

savetown2array:
	mov eax,numtowns*town2_size
	call savechunkheader

loadsavetown2array:
	xchg ecx,eax
	mov eax,[townarray2ofst]
	lea esi,[eax+townarray]
	call ebp

	or byte [extrachunksloaded1],LOADED_X1_TOWN2	// meaningless when saving
	ret


// End of extra chunk load/save/query functions

// find engines of each vehicle in all consists
findengines:
	pusha

	// first clear engine for all vehicles

	mov esi,[veharrayptr]
	mov edi,[veharrayendptr]
	push esi

.vehicleloop:
	mov word [esi+veh.engineidx],-1

.nextvehicle:
	sub esi,byte -vehiclesize
	cmp esi,edi
	jb short .vehicleloop

	// now find all engines and record them in their consists
	pop esi		// pop+push = mov esi,[veharrayptr]
	push esi

.vehicleloop2:
	cmp byte [esi+veh.class],0
	je .nextvehicle2

	cmp word [esi+veh.engineidx],byte -1
	jne .nextvehicle2

	// follow the chain of vehicles
	// (whether the current one is an engine or not)
	push esi
	mov ax,[esi+veh.idx]

.recordnext:
	mov [esi+veh.engineidx],ax
	movzx esi,word [esi+veh.nextunitidx]
	cmp si,byte -1
	je .nextvehicle2a
	shl esi,vehicleshift
	add esi,[veharrayptr]
	jmp short .recordnext

.nextvehicle2a:
	pop esi

.nextvehicle2:
	sub esi,byte -vehiclesize
	cmp esi,edi
	jb short .vehicleloop2

	// and finally set engineidx=idx for all vehicles which we
	// haven't touched yet (just to be safe)

	pop esi

.vehicleloop3:
	cmp word [esi+veh.engineidx],byte -1
	jne .nextvehicle3

	mov ax,[esi+veh.idx]
	mov [edi+veh.engineidx],ax

.nextvehicle3:
	sub esi,byte -vehiclesize
	cmp esi,edi
	jb short .vehicleloop3


	popa
	ret


// set the default save game file name
// if saved veh.mult is 0 or 1, it's "TRT00.SV1", otherwise "TRP00.SV1"
// safe:ebx,esi,edi,al,cl
// out:	edi=ptr to where filename is stored (only for the patch)
proc savedefaultname
	arg fn

	_enter

	mov edi,[%$fn]
	mov dword [edi],"TRT0"

	testflags uselargerarray
	jnc short .keepname

	cmp byte [vehicledatafactor],1
	jbe short .keepname

	mov byte [edi+2],'P'	// save as TRP??.SV1

.keepname:

	_ret
endproc ; savedefaultname



checkloadsuccess:
	pushf
	movzx eax,byte [loadreduced]
	test al,al
	jnz .reduced

	test byte [loadproblem],LOADPROBLEM_RCHUNKNOTLOADED
	jnz .rchunksnotloaded

	cmp dword [spriteerror],0
	je .done

	call grferror

.done:
	popf
	jnz short .dofarjmpnotret
	ret

.dofarjmpnotret:
	add esp,byte 4		// remove return address from stack
	jmp near $+0x200	// far jmp instead
ovar endofloadtarget,-4

.rchunksnotloaded:
	pusha
	mov bx,ourtext(rchunknotloaded1)
	mov dx,ourtext(rchunknotloaded2)
	jmp short .showerror

.reduced:
	pusha

	// show error popup
	mov esi,textrefstack
	mov [esi],eax			// store the original vehicle array multiplier
	mov eax,[loadremovedvehs]
	mov [esi+2],eax
	mov eax,[loadremovedsfxs]
	mov [esi+6],eax			// upper word irrelevant

	mov bx,ourtext(vehmulttoolow1)
	mov dx,ourtext(vehmulttoolow2)

.showerror:
	xor ax,ax
	xor cx,cx
	call dword [errorpopup]

	// pause the game
	mov bl,1
	mov esi,0x38
	call dword [actionhandler]

	popa
	jmp .done
; endp checkloadsuccess

// same as above but for loading title.dat
checktitleloadsuccess:
	cmp dword [spriteerror],0
	je .done

	call grferror

.done:
	mov ebx,6
	call [ebp+4]
	ret

grferror:	// error while loading graphics
	pusha
	xor esi,esi
	xchg esi,[spriteerror]	// reset error after showing message

	mov ax,[esi+spriteblock.cursprite]
	sub ax,[esi+spriteblock.firstsprite]
	inc ax		// first sprite is sprite #1
	mov esi,[esi+spriteblock.filenameptr]

	mov [textrefstack+2],ax

	mov [specialtext1],esi

	mov word [textrefstack],statictext(special1)
	mov bx,ourtext(grfloaderror)
	mov dx,[operrormsg2]
	xor ax,ax
	xor cx,cx
	call dword [errorpopup]
	popa
	ret



#if WINTTDX
// Fix schedule pointers if loading a savegame from TTDPatchW 1.7 and older
// in: ESI -> vehicle array, ECX = newvehicles, EDX = offset to add
bcfixcommandaddr:
.fixcommandloop:
	cmp	byte [esi],0
	jz	short .nextveh
	cmp	dword [esi+veh.scheduleptr],byte -1
	jz	short .nextveh
	cmp	dword [esi+veh.scheduleptr],edx
	jnc	short .nextveh

	// when called before saving, edx = -dsbase (negative)
	// so unsigned (commandindex >= edx) cannot happen;
	// when called after loading or saving, edx = dsbase
	// so (commandindex >= edx) indicates bug in the game just loaded
	// in which case commandindex is best left alone

	add	dword [esi+veh.scheduleptr],edx
.nextveh:
	sub	esi,byte -vehiclesize	//add	esi,vehiclesize
	loop	.fixcommandloop
	ret
; endp bcfixcommandaddr

// Before we reduce the vehicle array, we have to adjust schedule pointers;
// after that, we have to set them back so that TTD's adjustment code works
// (we cannot use TTD's code because the loaded array is larger)
// in:	EAX -> vehicle array, EBP -> vehicle array's end, EDX = 1 (adjust) or -1 (set back)
// safe:everything
adjustscheduleptrs:
	pusha
	imul edx,datastart
	add dword [scheduleheapfree],edx
	mov esi,eax
	mov ecx,ebp
	sub ecx,eax
	shr ecx,vehicleshift
	call bcfixcommandaddr		// note: the fix will work whether adjustscheduleptrs is called or not
	popa
	ret

#endif /* WINTTDX */


// Fix AI's vehicle pointers on load or save
// in:	EDX -> correction subfunction (adjaivehicleptrsload or adjaivehicleptrssave)
// preserves:ESI,EBP
adjaivehicleptrs:
	movzx ebx,byte [vehicledatafactor]
	imul ebx,oldveharraysize
	mov edi,[playerarrayptr]
	xor ecx,ecx
	mov cl,8

.companyloop:
	cmp word [edi],0
	je .next
	mov al,[edi+0x2bb]
	cmp al,2
	je .needadj
	cmp al,3
	je .needadj
	cmp al,4
	je .needadj
	cmp al,0x14
	jne .next

.needadj:
	mov eax,[edi+0x2c2]
	call edx
	mov [edi+0x2c2],eax

.next:
	add edi,player_size
	loop .companyloop
	ret

// Adjust after load or save:
adjaivehicleptrsload:
	cmp eax,byte -1
	je .done

	sub eax,oldveharray_abs
	cmp eax,ebx
	jae .bad
	test al,vehiclesize-1
	jnz .bad

	add eax,[veharrayptr]
	jmp short .done

.bad:
	or eax,byte -1
	cmp byte [edi+0x2bb],2
	je .done
	mov byte [edi+0x2bb],1

.done:
#if WINTTDX
	sub eax,datastart		// later adjusted back by WinTTDX
#endif
	ret

// Adjust before save:
adjaivehicleptrssave:
#if WINTTDX
	add eax,datastart
#endif
	cmp eax,byte -1
	je .done

	sub eax,[veharrayptr]
	add eax,oldveharray_abs

.done:
	ret
