// New mountain and curve handling

// this is to calculate the new speed when going up or downhill
mountain:
	push eax

	mov ah,cl
	mov cl,[esi+veh.tracktype]
	cmp word [esi+veh.vehtype],ROADVEHBASE
	jb .notroadveh
	mov cl,3

.notroadveh:
	add cl,cl
	mov al,[mountaintype]
	shr al,cl
	and al,3
	xchg cl,ah
	cmp al,3		// 3 = realistic
	je .realistic

	// dl has z pos of the previous square
	cmp dl,byte [esi+veh.zpos]
	je short .mountainexit
	ja short .mountainaccel

	cmp al,1
	jb .normal		// 0 = normal
	// je .fast		// 1 = faster (default case)
	ja .fullspeed		// 2 = full speed


.fast:		// faster speed: decelerate 6%
	mov dx,word [esi+veh.speed]
	shr dx,4
	sub word [esi+veh.speed],dx
	pop eax
	ret

.normal:	// normal speed: decelerate 25%
	mov dx,word [esi+veh.speed]
	shr dx,2
	sub word [esi+veh.speed],dx

.fullspeed:	// full speed
	pop eax
	ret


.mountainaccel:		// Accelerate downwards: 2 mph per time unit
	mov dx,word [esi+veh.speed]
	add dx,byte 2
	cmp dx,word [esi+veh.maxspeed]
	ja short .mountainexit
	mov word [esi+veh.speed],dx
.mountainexit:
	pop eax
	ret

// called when the engine of a consist moves in z direction
// with realistic acceleration
//
// in:	esi=vehicle
//	dl=zpos of previous square
// out:	zero flag according to replaced instructions
// safe:ax cx ebp
//
// this sets [esi+acceleration] to record the vertical motion
// the two upper bits (6&7) define the vertical motion:
//	+1 = uphill	(01xxxxxx)
//	 0 = flat	(00xxxxxx)
//	-1 = downhill	(11xxxxxx)
// the six lower bits are those of the sum of the x and y coordinates
// for motion detection
//
// for road vehicles, the sense of bits 6&7 is reversed
//
// This is achieved through keeping track of the last position of the vehicle
// and its vertical motion in [esi+veh.acceleration] which is encoded as
// follows:
//
// Bits
// 6,7	last vertical motion, either -1<<6 (0xC0) for downhill, 0 for flat
//	or +1<<6 (0x40) for uphill
// 4,5	amount of motion since vertical motion was recorded last
//	vertical motion is reset to 0 if no further vertical motion occurs
//	within two advances of the vehicle
// 2,3	least significant bits of last x position + bits 2,3 of last y position
// 0,1	least significant bits of last y position
//
// The vehicle movement can be detected by any change in bits 0 to 3.
//

.realistic:
	mov ah,[esi+veh.xpos]
	shl ah,2
	add ah,[esi+veh.ypos]
	and ah,0x0f

	cmp dl,[esi+veh.zpos]
	je .nochange

	// either up or downhill
	sbb al,al	// uphill=0, downhill=-1
	and al,1<<7	// uphill=0, downhill=-128
	add al,1<<6	// uphill=64, downhill=-64

	// now al = 1<<6 (0x40) for uphill, -1<<6 (0xC0) for downhill
	or al,ah
	jmp short .done

.nochange:
	// No change in z position. If there was a change in x or y,
	// that means the vehicle is not on a hill
	mov al,[esi+veh.acceleration]
	and al,(1<<4)-1
	cmp al,ah
	jz .nomotion	// no motion in x or y -> keep z direction

	mov al,[esi+veh.acceleration]
	and al,0xf0	// mask out amount of motion bits 4,5 and z dir bits 6,7
	or al,ah	// set bits 0 to 3
	add al,0x10
	test al,0x20
	jz .done	// not a lot of motion (z only changes every two x, y changes)

	// either x or y changed, so set z direction to zero (no hill)
	mov al,ah

.done:
	mov [esi+veh.acceleration],al

.nomotion:
	pop eax
	ret


// this determines the maximum speed in a curve for trains
curve:
	push eax
	mov ah,cl
	mov cl,[esi+veh.tracktype]
	add cl,cl
	mov al,[curvetype]
	shr al,cl
	and al,3
	mov cl,ah
	cmp al,3
	je .realistic

	and dh,7
	cmp dh,1
	je short .iswide
	cmp dh,7
	je short .iswide

.istight:		// a tight curve = 90 degree angle, always 1/2 speed
	shr word [esi+veh.speed],1

.realistic:
	pop eax
	ret

.iswide:		// a wide curve = 45 degree angle, use speed setting
	cmp al,1
	jb .normal		// 0 = normal
	ja .fullspeed		// 2 = full speed
				// 1 = faster

.faster:		// 7/8 speed
	mov ax,word [esi+veh.speed]
	shr ax,3	// subtract 1/8
	sub word [esi+veh.speed],ax
	pop eax
	ret

.normal:		// 3/4 speed
	mov ax,word [esi+veh.speed]
	shr ax,2	// subtract 1/4
	sub word [esi+veh.speed],ax

.fullspeed:		// full speed
	pop eax
	ret

; endp curve

// same as above but for road vehicles
// in:	esi=vehicle
// out:	(set speed)
// safe:ebp
rvcurve:
	push eax
	mov al,[curvetype]
	and al,3<<6
	jz .normal		// 0 = normal
	cmp al,2<<6
	jb .fast		// 1 = faster
	je .fullspeed		// 2 = full speed

				// 3 = realistic

	// limit to 3/4 of top speed
	movzx eax,word [esi+veh.maxspeed]
	lea eax,[eax+eax*2]
	shr eax,2

	// if the vehicle is entering a station, limit to 20 mph
	cmp byte [esi+veh.movementstat],0x20
	jb .limit
	cmp byte [esi+veh.movementstat],0x30
	jae .limit
	mov ax,64

.limit:
	cmp [esi+veh.speed],ax
	jbe .done
	mov [esi+veh.speed],ax

.done:
	pop eax
	ret


.fast:			// 7/8 speed
	mov ax,word [esi+veh.speed]
	shr ax,3	// subtract 1/8
	sub word [esi+veh.speed],ax
	pop eax
	ret

.normal:		// 3/4 speed
	mov ax,word [esi+veh.speed]
	shr ax,2	// subtract 1/4
	sub word [esi+veh.speed],ax

.fullspeed:		// full speed
	pop eax
	ret


// called when a road vehicle is accelerating
// in:	ax=current speed
//	esi=vehicle
// out:	eax=new speed (will be limited to maxspeed upon return)
// safe:ebx
rvaccelerate:
	push edx

	cwde	// clear high word of ax

	// calculate acceleration
	push eax

	call getrvweightandpower
	// now eax=power, ebx=total weight in 1/16 tons

	// acceleration = force / mass = (power/speed) / mass = power / (speed * mass)
	// (assuming perfect gear ratios at each speed, which is
	//  something of an oversimplification but close enough)
	// below 20 mph, acceleration is constant and maximal

	shl eax,16		// for more accuracy

	mov edx,[esp]		// [esp] = current speed
	test edx,edx
	jz .tooslow

	imul ebx,edx
	xor edx,edx

.tooslow:
	idiv ebx

	// now eax=acceleration
	mov ebx,eax
	shr ebx,1

	// add a little friction
	// kinetic friction (constant 2), rolling friction (speed/64)
	// and air resistance (speed^2/16000)
	mov eax,[esp]
	mov edx,eax
	imul edx,eax

	shr eax,6
	shr edx,14

	// could put variable coefficients of friction here to
	// support *really* fast trucks

	lea eax,[eax+edx+2]
	sub ebx,eax

	mov eax,345	// maximum tractive acceleration = 0.9 g or so
	cmp ebx,eax
	jbe short .notslipping

	mov ebx,eax

.notslipping:

	// adjust for gravity up/downhill if necessary
	movsx edx,byte [esi+veh.acceleration]
	and edx,0-(1<<6)

	// 96 is an arbitrary value for the tangential component of
	// the acceleration of gravity

	add ebx,edx
	sar edx,1
	add ebx,edx

	pop eax


	cmp byte [esi+0x66],0
	je .nottwice

	add ebx,ebx	// double acceleration sometimes?  That's what TTD does.

.nottwice:
	mov dl,bl
	sar ebx,8

	add [esi+veh.speedfract],dl
	adc eax,ebx

	jns .notneg

	xor eax,eax

.notneg:
	pop edx
	ret

; endp rvaccelerate

	uvard lasttractiveeffort
	uvard lastaccel


// same as above but for trains
// in:	esi=vehicle
// out:	eax=new speed (will be limited to max speed later)
// safe:ebx
proc trainaccelerate
	local adhweight,totweight,inclforces,maxte

	_enter

	push ecx
	push edx

		// count weight and incline forces of all engines+waggons
	xor ebx,ebx
	mov [%$adhweight],ebx
	mov [%$totweight],ebx
	mov [%$inclforces],ebx

	mov cl,[esi+veh.zpos]

	push esi

.next:
	movsx ebx,word [esi+veh.fullweight]
	test ebx,ebx
	js .ok

	call calcaccel.storeweight
	jmp .next	// get it again

.ok:
	movzx eax,byte [esi+veh.vehtype]

	bt [isengine],eax
	jnc .waggon

	sub [%$adhweight],ebx

.waggon:
	sub [%$totweight],ebx

	mov ch,[esi+veh.zpos]
	cmp cl,ch
	je .noslope
	ja .uphill

	neg ebx

.uphill:
	shl ebx,6
	add [%$inclforces],ebx

	mov cl,ch
.noslope:

	movzx esi,word [esi+veh.nextunitidx]
	cmp si,byte -1
	je .haveall
	shl esi,vehicleshift
	add esi,[veharrayptr]
	jmp .next


.haveall:
	pop esi

	// add incline force for the engine
	movsx eax,byte [esi+veh.acceleration]
	movsx ebx,word [esi+veh.fullweight]

	and eax,byte -64
	imul eax,ebx

	sub [%$inclforces],eax

	mov eax,[esi+veh.realpower]
	test eax,eax
	jnz .gotpower

	movzx edx,byte [esi+veh.vehtype]
	mov ebx,[enginepowerstable]
	movzx eax,word [ebx+edx*2]

.gotpower:
		// do conversion from hp/(mph/1.6) to N
		// hp/(mph/1.6) = 2.669 kN

	mov ebx,683		// 2.669*256
	mul ebx

	movzx ebx,word [esi+veh.speed]
	test ebx,ebx
	jz .tooslow

	div ebx

.tooslow:
	// now eax=force of engine [kN/256]

	mov edx,[%$adhweight]
	shl edx,8
	lea edx,[edx+edx*2]	// now edx: maximum TE = adh.mass*10*0.3 [kN/256]

	cmp eax,edx
	jbe .belowmax

	mov eax,edx

.belowmax:
	mov [%$maxte],edx
	// now eax=tractive effort [kN/256]

	mov [lasttractiveeffort],eax

	// calculate coefficient of air resistance
	// depending on max speed
	movzx edx,word [esi+veh.maxspeed]
	bsr ecx,edx

	shr edx,cl
	jnc .notodd

	mov ch,1
	inc cl

.notodd:
	sub cl,4
	jg .factorok

	mov cl,1
	mov ch,0

.factorok:
	cmp byte [esi+veh.movementstat],0x40
	jne .nottunnel

	dec cl		// cl was at least 1

.nottunnel:

	// now ch, cl give the following divisors (using bit shifts):
	// top speed: <32 64 96 128 192 256 384 512 768 1024 ...
	// divisor:     2  4  6   8  12  16  24  32  48   64 ...
	// (half as much in tunnels = twice the air resistance)

	mov edx,ebx

	imul edx,edx

	test ch,ch
	jz .notadded
	lea edx,[edx+edx*2]

.notadded:
	shr edx,cl		// edx: air resistance = c2*v^2

	add ebx,50
	shr ebx,4		// ebx: c0+c1*v = friction/m (=frictional accel.)

	sub eax,edx

	mov edx,[%$inclforces]	// edx = mass*9.8*sin(theta)/2 [kN/256] (theta=5%)
	add edx,edx
	add eax,edx		// eax = TE + inclforce

	imul edx,byte -6	// edx = -inclforce*6
	lea edx,[edx+eax*4]	// edx = (TE - inclforce/2)*4

	cmp edx,[%$maxte]
	jle .nosmoke

		// show smoke if TE + inclforce/2 > maxTE/4
	or byte [esi+veh.modflags],1 << MOD_SHOWSMOKE

.nosmoke:

	// now eax=net force (except for static/rolling friction
	// which is in ebx as an acceleration)

	cdq		// sign extend edx
	idiv dword [%$totweight]

	sub eax,ebx		// subtract friction (avoid multiplying and dividing by weight)

	// now eax=acceleration [m/s^2/256]

	mov [lastaccel],eax

	add eax,eax		// convert to more useful numbers

	movzx ebx,word [esi+veh.speed]

	// calculate new speed

	mov dl,al
	sar eax,8

	add byte [esi+veh.speedfract],dl
	adc eax,ebx

	cmp ebx,eax
	jl .nottoosmall

	cmp eax,2
	jge .nottoosmall

	mov eax,2	// don't let speed drop below 1.25 mph

.nottoosmall:
	pop edx
	pop ecx
	_ret

endproc trainaccelerate


	// called to calculate the new speed after acceleration
	//
	// in:	esi=vehicle
	// out:	(set esi.speed and esi.subspeed), ax=speed
	// safe:eax,ebx
calcspeed:
	push ecx
	mov cl,[esi+veh.tracktype]
	add cl,cl
	mov al,[mountaintype]
	shr al,cl
	and al,3
	cmp al,3
	jne .notrealisticaccel
	call trainaccelerate
	jmp short .limittomaxspeed

.notrealisticaccel:
	movzx eax,word [esi+veh.fullaccel]
	shl eax,2
	jnz short .havefullaccel
	movzx eax,byte [esi+veh.acceleration]
	shl eax,2
.havefullaccel:
	mov bl,al
	shr eax,8
	add byte [esi+veh.speedfract],bl
	movzx ebx,word [esi+veh.speed]
	adc eax,ebx

.limittomaxspeed:
	mov bl,[curvetype]
	shr bl,cl		// cl is still 2*tracktype
	and bl,3
	cmp bl,3
	movzx ebx,word [esi+veh.maxspeed]
	jne .gotspeed

	push esi

	mov cl,0

.next:
	mov ch,[esi+veh.direction]

	movzx esi,word [esi+veh.nextunitidx]
	cmp si,-1
	je .done
	shl esi,vehicleshift
	add esi,[veharrayptr]

	sub ch,[esi+veh.direction]
	je .next

	inc cl

	and ch,7
	cmp ch,1
	je .next	// wide curve
	cmp ch,7
	je .next

	add cl,2	// narrow curve -> counts as 3
	jmp .next

.done:
	pop esi

	// now cl=total number of turns the entire train is making

	// for each turn, subtract 1/16, up to 8/16

	cmp cl,8
	jbe .ok

	mov cl,8

.ok:
	movzx ecx,cl
	imul ecx,ebx
	shr ecx,4
	sub ebx,ecx

.gotspeed:
	cmp eax,ebx
	jbe short .nottoofast
	mov eax,ebx

.nottoofast:
	mov [esi+veh.speed],ax
	pop ecx
	ret
; endp calcspeed
