//
// This file is part of TTDPatch
// Copyright (C) 1999-2001 by Josef Drexler
//
// C++ to C conversion by Marcin Grzegorczyk
//
// switches.c: defines functions dealing with the switches
//

#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <ctype.h>

#ifdef __BORLANDC__
#	include <conio.h>
#else
#	define getch() getc(stdin)
#	define stricmp strcasecmp
#endif

#include "codepage.h"
#include "error.h"

#define IS_SWITCHES_CPP
#include "switches.h"
#include "language.h"

#if WINTTDX
#	include "versionw.h"
#else
#	include "versiond.h"
#endif


int showswitches = 0;
//int writeverfile = 0;
u16 startyear = 1950;
char ttdoptions[128+1024*WINTTDX];
int cfgfilespecified = 0;
int forcerebuildovl = 0;
int mcparam[2] = {0, 0};

extern char *patchedfilename;

#define DEFAULTCFGFILE "ttdpatch.cfg"


void givehelp(void)
{
  int i, total;

  printf("TTDPatch - ");
  printf(langtext[LANG_COMMANDLINEHELP], patchedfilename);

  // count halflines
  for (total=0; halflines[total]; total++);

  // show them in two-column format
  for (i=0; i<total/2; i++)
	printf("%-38s %c %-38s\n", halflines[i], langtext[LANG_SWTABLEVERCHAR][0], halflines[i + (total + 1)/2]);
  if (total % 2)
	printf("%-38s %c\n", halflines[(total + 1)/2 - 1], langtext[LANG_SWTABLEVERCHAR][0]);

  printf("%s%s%s",
	langtext[LANG_FULLSWITCHES],
	"Copyright (C) 1999-2002 by Josef Drexler.  ",
	langtext[LANG_HELPTRAILER]);
  exit(0);
}


void copyflagdata(void)
{
  // copy switch variables we handle specially to the flag data
  int i, p;

  flags->data.startyear = (u8)(startyear - 1920);

  for (p=0; p<2; p++) {
	flags->data.mctype[p] = 0;
	if (getf(usenewcurves+p))
	  for (i=0; i<4; i++) {
		flags->data.mctype[p] |= ((mcparam[p] >> ((3-i)*4)) & 3) << (i*2);
	  }
  }
}



#define OBSOLETE ((void*)-1L)

int radix[4] = { 0, 8, 10, 16 };

#define YESNO(ch, txt, comment, sw) \
	{ ch, txt, comment, sw,  0, 0, {-1, -1, -1}, 0, NULL }

#define SPCL(ch, txt, comment, var) \
	{ ch, txt, comment, -1,  0, 2, {-1, -1, -1}, 0, var }

#define RANGE(ch, txt, comment, sw, radix, varsize, var, low, high, default) \
	{ ch, txt, comment, sw, radix, varsize, {low, high, default}, 0, var }


#define noswitch -2
#define FLAGDATA(var) ( (void _fptr*) ( (s32) ( offsetof(paramset, data.var) - offsetof(paramset, data)) ) )
#include "sw_lists.h"
#undef noswitch

int overridesconfigfile = 0;

int readcfgfile(const char *filename);
int writecfgfile(const char *filename);


// Parameters for the on/off switches, case is ignored.  *MUST* be unique.
// Number of entries not limited.  End each list with a NULL entry.
// The first value is what is printed by the -W switch.
// e.g. "presignals yes" or "presignals off" etc.

const char *const switchonofftext[] =
	{ "on", "yes", "y", NULL,
	  "off", "no", "n", NULL };

const char *switchofftext = "off";

void _fptr *getswitchvarptr(int switchid)
{
  void _fptr *ptr = switches[switchid].var;

  if ( (u32) ptr < sizeof *flags )
	ptr = (void _fptr*) ( (char _fptr*) &(flags->data) + (s32) ptr);

  return (void _fptr*) ptr;
}

void setswitchvar(int switchid, s32 value)
{
  void _fptr *ptr = getswitchvarptr(switchid);

  if (switches[switchid].radix & 4)
	value = ~value;

  switch (switches[switchid].varsize) {
	case 0:
		*( (u8 _fptr *) ptr) = value;
		break;
	case 1:
		*( (u16 _fptr *) ptr) = value;
		break;
	case 2:
		*( (s16 _fptr *) ptr) = value;
		break;
	case 3:
		*( (s32 _fptr *) ptr) = value;
		break;
	case 4:
		*( (s8 _fptr *) ptr) = value;
		break;
  }
}

s32 getswitchvar(int switchid)
{
  s32 value, mask;
  void _fptr *ptr = getswitchvarptr(switchid);

  switch (switches[switchid].varsize) {
	case 0:
		value = *( (u8 _fptr *) ptr);
		mask = 0xff;
		break;
	case 1:
		value = *( (u16 _fptr *) ptr);
		mask = 0xffff;
		break;
	case 2:
		value = *( (s16 _fptr *) ptr);
		mask = 0xffff;
		break;
	default:				// case 3 really
		value = *( (s32 _fptr *) ptr);
		mask = 0xffffffff;
		break;
	case 4:
		value = *( (s8 _fptr *) ptr);
		mask = 0xff;
		break;
  }

  if (switches[switchid].radix & 4)
	value = (~value & mask);

  return value;
}


void initvalues(int preferred)
{
  int i;

  // first set default values for all switches as if they were "on"

  for (i=0; switches[i].cmdline; i++) {
	if ( (switches[i].range[0] != -1) || (switches[i].range[1] != -1) )
		setswitchvar(i, switches[i].range[2]);
  }

  // if electrifiedrailway is on (which it is now), the default
  // value for unimaglevmode isn't valid
  flags->data.unimaglevmode = 1;

  if (!preferred) {	// now set default values for "off" switches,
			// i.e. as if the switch wasn't on
			// where it should be different from the "on" default
	flags->data.newvehcount[0] = 80;
	flags->data.newvehcount[1] = 80;
	flags->data.newvehcount[2] = 50;
	flags->data.newvehcount[3] = 40;
	mcparam[0] = 0;
	mcparam[1] = 0;
	flags->data.vehicledatafactor = 1;
	flags->data.newstationspread = 12;
	flags->data.newservint = 180;
	flags->data.planecrashctrl = 0;
	flags->data.multihdspeedup = 0;
	flags->data.signalwaittimes[0] = 41;
	flags->data.signalwaittimes[1] = 15;
	flags->data.unimaglevmode = 0;
	flags->data.newbridgespeedpc = 50;
	startyear = 1950;
	flags->data.redpopuptime = 4;
	flags->data.bigtownfreq = 0;
	flags->data.townsizelimit = 20;
	flags->data.treeplantmode = 0;
	flags->data.towngrowthratemode = 0;
	flags->data.townminpopulationsnow = 0;		// important!
	flags->data.townminpopulationdesert = 0;	// important!

  }
}

void allswitches(int reallyall, int swon)
{
  int last, i;

  if (
	reallyall
#if DEBUG
	|| 1		// for debug versions, -a turns on all switches
#endif
     )
	last = lastbit;
  else
	last = lastbitused;

  for (i=0; i<=last; i++)
	if ( (i != lowmemory) )
		setf(i, swon);

/*
  if (reallyall) {
	// so that these switches properly get their versiondata
	setf(win2k, swon);
  }
*/

  initvalues(swon);
  copyflagdata();

}


int setreallyspecial(int switchid, int swon, const char *cfgpar, int onlycheck)
{
  int parused = 0;

  switch (switches[switchid].cmdline) {
	case 'a':
		if (!onlycheck) allswitches(0, swon);
		break;
	case 'h':
	case '?':
		if (!onlycheck) givehelp();
		break;
	case 'C':
		if (!onlycheck)
			readcfgfile(cfgpar);
		else
			overridesconfigfile = 1;
		parused = 1;
		break;
	case 'W':
		if (!onlycheck)
			writecfgfile(cfgpar);
		parused = 1;
		break;
	case 1:		// CD Path
		if (!onlycheck) {
			if (cfgpar && (strlen(cfgpar) > 0))
				strcpy(ttdoptions, cfgpar);
			else
				strcpy(ttdoptions, "");
		}
		parused = 1;
		break;
  }
  return parused;
}

int lastswitchorder = 0;

int setswitch(int switchid, const char *cfgpar, int swon, int onlycheck)
{
  int parused = 0;
  char *endptr;
  s32 parvalue;

  if (!onlycheck && !debug_flags.switchorder)
	switches[switchid].order = ++lastswitchorder;

  if (swon < 0) {	// not yet determined
	swon = 1;	// switch given, implies default "on"
	if (cfgpar &&
	    (switches[switchid].cmdline != 1) &&	// ignore CDPath
	    (switches[switchid].cmdline != 'W') &&	// and -W / writecfg
	    (switches[switchid].range[0] == -1) &&
	    (switches[switchid].range[1] == -1) ) {
		// check if a non-ranged switch has a value
		parvalue = strtol(cfgpar, &endptr, 0);
		if (*endptr == 0)
			swon = (parvalue != 0);	// if it's zero turn the switch off
		else {
			warning(langtext[LANG_UNKNOWNSTATE], cfgpar);
			swon = 0;
		}
	}
  }

  if (switches[switchid].bit == -1) {	// Special switches
	if (switches[switchid].var == NULL) {	// really special
		parused = setreallyspecial(switchid, swon, cfgpar, onlycheck);
	} else if (switches[switchid].var == OBSOLETE) {
		int ch;
			// obsolete
		ch = switches[switchid].cmdline;
		if ( ((ch & 0xff) < 32) || ((ch & 0xff) > 128) )
			ch = 0;

		if (!onlycheck) warning(langtext[LANG_SWITCHOBSOLETE],
			switches[switchid].cfgcmd, dchartostr(ch));
	} else {	// with a variable to set
		if (!onlycheck)
			setswitchvar(switchid, swon);
	}
  } else if (!onlycheck) {	// not special
	if ( (switches[switchid].range[0] != -1) ||
	     (switches[switchid].range[1] != -1) ) {	// with a range (value)
		if (cfgpar == NULL || *cfgpar == 0)
			parvalue = switches[switchid].range[2];	//default
		else {
			if (*cfgpar == '#') { // We have bin format
			cfgpar++;
			parvalue = strtol(cfgpar, &endptr, 2);
				
			} else { 
			parvalue = strtol(cfgpar, &endptr,
					radix[switches[switchid].radix & 3]);
			
			}
			if (*endptr == 0) {
				int offrange = -1;
				parused = 1;

				if ( (parvalue < switches[switchid].range[0]) )
					offrange = 0;
				else if ( (parvalue > switches[switchid].range[1]) )
					offrange = 1;

				if (offrange >= 0) switch (parvalue) {
					default:
						// bring the value into bounds
						parvalue = switches[switchid].range[offrange];
						break;
					case 1:
						// value is 1 and not in range, assume it means "on"
						parvalue = switches[switchid].range[2];
						break;
					case 0:
						swon = 0;	// value is 0 and not in range, turn off the switch
				}
			}
			else
				parvalue = switches[switchid].range[2];
		}

		setswitchvar(switchid, parvalue);
	}

	if (switches[switchid].bit >= 0)
		setf(switches[switchid].bit, swon);
  }
  return parused;
}

// command line is parsed twice, first to only check if there is a '-C' switch
// second pass is for real
int processswitch(int switchchar, const char *cfgswline, int swon, int onlycheck)
{

  int i, k, l, switchid = -1;
  const char *cfgcmd, *cfgpar;

  if ( (switchchar == 0) && (cfgswline != NULL) ) {	// cfg entry
  	char *cfgline = (char *)cfgswline;		// modifiable (in a buffer)

	// syntax is "name *[:=]? *value", value may be omitted.

	i = strcspn(cfgline, " :=");
	if (cfgline[i])
		cfgline[i++] = 0;
	cfgcmd = cfgline;

	i += strspn(&(cfgline[i]), " :=");
	cfgpar = &(cfgline[i]);

	if (*cfgpar == 0)	// no parameter
		swon = 1;	// -> "yes" default
	else {
		swon = -1;	// not yet determined
		l = 1;		// first list is "on" list

		for (k=0; l >= 0; k++)
			if (switchonofftext[k]) {
				if (stricmp(cfgpar, switchonofftext[k]) == 0) {
					swon = l;
					break;
				}
			} else {
				if (l--) switchofftext = switchonofftext[k + 1];
			}
	}
  } else {
	cfgpar = cfgswline;	// parameter
	cfgswline = NULL;
	cfgcmd = NULL;
  }

  for (i=0; switches[i].cmdline; i++) {
	if (cfgswline == NULL) {
		if (switches[i].cmdline == switchchar) {
			switchid = i;
			break;
		}
	} else {
		if (switches[i].cfgcmd)
			if (stricmp(cfgcmd, switches[i].cfgcmd) == 0) {
				switchid = i;
				break;
			}
	}
  }

  if (switchid == -1)
	return 0;

  return 1 + setswitch(switchid, cfgpar, swon, onlycheck);
}

int readcfgfile(const char *filename)
{
#define CFGLINEMAXLEN 256
  char cfgline[CFGLINEMAXLEN + 2], cfglineorg[CFGLINEMAXLEN + 2];
  int linetoolong, linepos;

  FILE *cfgfile;

  if (filename == NULL) filename = "";

  if (strlen(filename) < 1) {
	printf(langtext[LANG_CFGFILENOTFOUND], filename);
	return 0;
  }

  cfgfile = fopen(filename, "rt");
  if (!cfgfile) {
	if (strcmp(filename, DEFAULTCFGFILE))
		printf(langtext[LANG_CFGFILENOTFOUND], filename);
	return 0;
  }

  while (!feof(cfgfile)) {
	memset(cfgline, 0, CFGLINEMAXLEN + 1);
	fgets(cfgline, CFGLINEMAXLEN, cfgfile);

	linetoolong = (strchr(cfgline, '\n') == NULL) && (!feof(cfgfile));

	if (linetoolong)
		fscanf(cfgfile, "%*[^\n]");	// skip to end of line

	for (linepos = 0; isspace(cfgline[linepos]); linepos++);

	if (cfgline[linepos] == 0)
		continue;	// skip empty lines

	if (isalpha(cfgline[linepos])) {	// all lines starting with a-z are options
		if (linetoolong)
			warning(langtext[LANG_CFGLINETOOLONG]);

		{
		  char *eol = strchr(cfgline, '\n');
		  if (eol) *eol = 0;
		}
		strcpy(cfglineorg, cfgline);

		if (!processswitch(0, &(cfgline[linepos]), 0, 0))
			warning(langtext[LANG_UNKNOWNCFGLINE], cfglineorg);
	}
  }

  fclose(cfgfile);

  return 1;
}

void writereallyspecial(int switchid, const char **const formatstring, s32 *parvalue)
{
  switch (switches[switchid].cmdline) {
	case 155:	// CD Path
		if (strlen(ttdoptions) > 0)
			*parvalue = (s32) ttdoptions;
		else
			*parvalue = (s32) "";
		*formatstring = "%s %s";
		break;
  }
}

void writerangedswitch(int switchid, const char **const formatstring, s32 *parvalue)
{
  if ((switches[switchid].bit >= 0) && !getf(switches[switchid].bit)) {
	*formatstring = "%s %s";
	*parvalue = (s32) switchofftext;
  } else {
	tempstr[0] = 0;
	*formatstring = tempstr;
	switch (switches[switchid].radix & 3) {
		case 1:
			strcat(tempstr, "%s %lo");
			break;
		case 3:
			strcat(tempstr, "%s %lx");
			break;
		default:
			strcat(tempstr, "%s %ld");
			break;
	}
	*parvalue = getswitchvar(switchid);
  }
}

void writeswitch(FILE *cfgfile, int switchid)
{
  s32 parvalue;
  const char *formatstring = "%s %ld";

  if (switches[switchid].bit == -1) {		// special
	if (switches[switchid].var == NULL) {		// really special
		writereallyspecial(switchid, &formatstring, &parvalue);
	} else if (switches[switchid].var == OBSOLETE) {	// obsolete
		return;
	} else {	// has a var
		parvalue = *( (int*) switches[switchid].var);
		if ( (parvalue >= 0) && (parvalue <= 1) ) {
			formatstring = "%s %s";
			if (parvalue)
				parvalue = (s32) switchonofftext[0];
			else
				parvalue = (s32) switchofftext;
		}
	}
  } else {	// not special
	if ( (switches[switchid].range[0] != -1) ||
	     (switches[switchid].range[1] != -1) ) {	// ranged (value)
		writerangedswitch(switchid, &formatstring, &parvalue);
	} else {	// just on/off
		formatstring = "%s %s";
		#if DEBUG
		if (switches[switchid].bit < 0)
			error("** %s: on/off switch with no flag\n", switches[switchid].cfgcmd); 
		#endif
		if (getf(switches[switchid].bit))
			parvalue = (s32) switchonofftext[0];
		else
			parvalue = (s32) switchofftext;
	}
  }

  fprintf(cfgfile, formatstring, switches[switchid].cfgcmd, parvalue);
}

char *dchartostr(int ch)
{
  static char dcharstr[sizeof(int) + 1];

  dcharstr[0] = firstchar(ch);
  dcharstr[1] = secondchar(ch);

  return dcharstr;
}

int writecfgfile(const char *filename)
{
  FILE *cfgfile;
  int switchid;
  int i, newsep;
  int *list;

  if (filename == NULL) filename = "";

  if (strlen(filename) < 1) {
	printf(langtext[LANG_CFGFILENOTFOUND], filename);
	return 0;
  }

  cfgfile = fopen(filename, "wt");
  if (!cfgfile) {
	printf(langtext[LANG_CFGFILENOTWRITABLE], filename);
	return 0;
  }

  fprintf(cfgfile, langcfg(CFG_INTRO), TTDPATCHVERSION);

  // build list of switches in the right order
  i = lastswitchorder;
  list = calloc(i + sizeof(switches)/sizeof(switches[0]) + 2, sizeof(int));
  if (list) {
	for (switchid=0; switches[switchid].cmdline; switchid++)
		if (switches[switchid].order)
			list[switches[switchid].order-1] = switchid+1;

	list[i++] = -1;		// end of old; beginning of new switches

	for (switchid=0; switches[switchid].cmdline; switchid++)
		if (!switches[switchid].order)
			list[i++] = switchid+1;

	list[i++] = -2;		// end of list
  }

  newsep = 0;
  for (i=0; ; i++) {
	if (list) {
		switchid = list[i]-1;

		if (switchid == -2)
			newsep = 1;
		else if (switchid == -3)
			break;

		if (switchid < 0)
			continue;
	} else {
		switchid = i;
		if (!switches[switchid].cmdline)
			break;
	}

	if (switches[switchid].comment) {
		if (newsep) {
			if (!debug_flags.switchorder)
				fprintf(cfgfile, "\n\n\n" CFG_COMMENT "%s\n",
					langcfg(CFG_NEWSWITCHINTRO));
			newsep = 0;
		}

		fprintf(cfgfile, "\n\n" CFG_COMMENT);
		// Note -- CFG_COMMENT uses ASCII, so no conversion is needed
		if (switches[switchid].cmdline & 128) {
			// no command line switch
			fprintf(cfgfile, langcfg(switches[switchid].comment),
				switches[switchid].cfgcmd,
				switches[switchid].range[0],
				switches[switchid].range[1],
				switches[switchid].range[2]);
		} else {
			fprintf(cfgfile, langcfg(switches[switchid].comment),
				switches[switchid].cfgcmd,
				dchartostr(switches[switchid].cmdline),
				switches[switchid].range[0],
				switches[switchid].range[1],
				switches[switchid].range[2]);
		}
		fprintf(cfgfile, "\n");

		writeswitch(cfgfile, switchid);
	}
  }

  fprintf(cfgfile, "\n");
  fclose(cfgfile);

  if (list) free(list);

  return 1;
}

#if LINTTDX
/* We don't handle -W well and we are compiled w/ ttdpatch by default so user
 * should at least explicitly confirm by -a that he wants to use it. */
const char *defaultcmdline[] = { NULL };
#else
const char *defaultcmdline[] = { NULL, "-a", "-W", DEFAULTCFGFILE };
#endif

void commandline(int argc, const char *const *argv)
{
  int i, par, swon, p, parused;
  const char *s, *switchpar;
  int switchchar;
  int onlycheck;

  memset(&flags->data, 0, sizeof(flags->data));
  initvalues(0);

  for (i=0; i<nflagssw; i++)
	flags->flags[i] = 0;

  alwaysrun = 0;

  for (onlycheck = 1; onlycheck >= 0; onlycheck--) {

	// "real" run, and there is no '-C' switch -> source default .cfg file
     if ( (!onlycheck) && (!overridesconfigfile) && (debug_flags.readcfg >= 0) )
	if (!readcfgfile(DEFAULTCFGFILE)) {
		// ttdpatch.cfg doesn't exist either; make a default
		// if we have *no* cmd.line options at all
		if (argc < 2 && !debug_flags.runcmdline) {
			argc = sizeof(defaultcmdline) / sizeof(defaultcmdline[0]);
			argv = defaultcmdline;
		}
	}

     for (par=1; par < argc; par++) {
	s = argv[par];

	if ( ((s[0] != '-') && (s[0] != '/')) || (debug_flags.runcmdline > 0) ) {
		if (!onlycheck) {	// handle non-option strings, but only in the "real" run
			size_t len = strlen(ttdoptions);
			if (len < sizeof(ttdoptions) - 2) {
				if (len > 0)
					strcat(ttdoptions, " ");
				strncat(ttdoptions, s, sizeof(ttdoptions) - len - 2);
			}
		}
		continue;
	}

	switchpar = argv[par + 1];	// NULL if par==argc-1
	parused = 0;

	if ((s[0] == '-') && (s[1] == '-')) {	// long switch
		char longswitch[CFGLINEMAXLEN+1];
		size_t longswitchlen = strlen(s + 2);

		if (strlen(s + 2) >= CFGLINEMAXLEN)
			error(langtext[LANG_UNKNOWNSWITCH], s[0], s + 1);

		strcpy(longswitch, s + 2);
		if (!longswitch[strcspn(longswitch, " :=")] && switchpar) {
			strcat(longswitch, "=");
			strncat(longswitch, switchpar, CFGLINEMAXLEN - longswitchlen - 1);
			parused = 1;
		}

		if (!processswitch(0, longswitch, 0, onlycheck))
			error(langtext[LANG_UNKNOWNSWITCH], s[0], s + 1);

	} else for (p=1; s[p]; p++) {
		switchchar = s[p];
		if ( (switchchar == 'X') || (switchchar == 'Y') ) {
			p++;
			if (!s[p])
				error(langtext[LANG_UNKNOWNSWITCH], s[0], dchartostr(switchchar));
			switchchar = maketwochars(switchchar, s[p]);
		}

		if (onlycheck && (switchchar != 'C') )
			continue;

		swon = 1;
		if (s[p + 1] == '-') {
			swon = 0;
			p++;
		}

		i = processswitch(switchchar, switchpar, swon, onlycheck);
		if (!i)
			error(langtext[LANG_UNKNOWNSWITCH], s[0], dchartostr(switchchar));

		if (i == 2)
			parused = 1;
	}

	if (parused)
		par++;
     }
  }

  if (debug_flags.useversioninfo > 0)
	setf1(recordversiondata);

  if (debug_flags.terminatettd)
	setf1(onlygetversiondata);

  if (debug_flags.noshowleds < 0)
	setf1(dontshowkbdleds);

  if (flags->data.vehicledatafactor <= 1)
	clearf(uselargerarray);

  // check dependencies between switches
  if (debug_flags.chkswitchdep >= 0) {
    if (!getf(uselargerarray)) {
	flags->data.vehicledatafactor = 1;
    }

    #if WINTTDX
	clearf(lowmemory);    
    #endif

    if (getf(gradualloading))
	setf1(improvedloadtimes);

    if (!getf(autorenew))
	clearf(forceautorenew);

    if (getf(electrifiedrail)) {
	setf1(unifiedmaglev);
	if (flags->data.unimaglevmode != 2)
		flags->data.unimaglevmode = 1;
    }

    if (getf(largertowns) && !getf(newtowngrowthfactor)) {
	setf1(newtowngrowthfactor);
	flags->data.townsizelimit = 80;
    }

    if (getf(recordversiondata)) {
	setf1(canmodifygraphics);
	setf1(enhancedkbdhandler);
    }

    if (!getf(newtowngrowthrate)) {
	flags->data.towngrowthratemode = 0;
    }

    if (getf(startyear) && (startyear < 1930)) {
	setf1(generalfixes);
	// flags->data.miscmodsflags &= ~2;	// no longer needed
    }

  }

  copyflagdata();

}


// Switches display

	      // how many to show
#define notshowswitches 3	// setsignal{1,2}waittime, win2k or lowmemory
#define shownswitches ((lastbit)+1 - notshowswitches)
#define lines ((shownswitches + 1) / 2)

// show either win2k or lowmemory depending on the version
#if WINTTDX
#	define VSW win2k
#else
#	define VSW lowmemory
#endif

	// fill with -1 if it's an even number at the end
s8 switchorder[lines*2] =				// display index
	{  1,  2,  3,  6,  7, 49,  8, 13,		// 0..7
	  14,  0, 11, 16, 44, 18, 19,  4,  		// 8..15
	   5,  9, 10, 20, 21, 22, 23, 24,		// 16..23
	  12, 15, 25, 26, 27, 28, 30, 39, 		// 24..31
	  32, 33, 34, 35, 36, 37, 38,VSW, 		// 32..39
	  17, 41, 42, 43, 40, 45, 62, 48, 		// 40..47
	  50, 51, 52, 53, 54, 55, 56, 57, 		// 48..55
	  58, 59, 60, 61, 63, 64, 65, 66,		// 56..63
	  67, 68, 69, 70, 71, 72, 73, 74, 		// 64..71
	  75, 76, 77, 78, 79, 80, 81, 82,		// 72..77
	  83, -1,					// 78..79
	};


void showoneswitch(int bit, int column)
{
  char flag = getf(bit);
  char line[80], *lineend = line;
  int i, switchid;
  s32 value = -1;		// initialized to make gcc happy
  const char *midtext = NULL;

  // find the switch in switches[]
  for (i=0;; i++) {
	if (!switches[i].cmdline) {
	  #if DEBUG
		printf("[%d]: no switch\n", bit);
	  #endif
		return;
	}
	if (switches[i].bit == bit) {
		switchid = i;
		break;
	}
  }

  if ( (switches[switchid].range[0] != -1) || (switches[switchid].range[1] != -1) ) {
	value = getswitchvar(switchid);
	midtext = switchnames[bit*2+1];
  }

  lineend += sprintf(line, " %c ",
			flag?langtext[LANG_SWTABLEVERCHAR][1]
			    :langtext[LANG_SWTABLEVERCHAR][2]);

  lineend += sprintf(lineend, switchnames[bit*2]);

  if (flag && midtext)
	sprintf(lineend, midtext, value);

  printf("%s", line);

  if (column == 1)
	printf("%*c", 40-strlen(line), langtext[LANG_SWTABLEVERCHAR][0]);
  else if ( (column == 2) && (strlen(line) < 40) )
	printf("\n");
}

void showwaittime(const char *head, int value)
{
  printf("  %s", head);

  if (value == 255)
	printf(langtext[LANG_INFINITETIME]);
  else
	printf(langtext[LANG_TIMEDAYS], value);
}

void showtheswitches(void (*ensureconsize)(int))
{
  int i;

  printf(langtext[LANG_SWWAITFORKEY]);
  fflush(stdout);
#if LINTTDX
  getchar(); /* XXX */
#else
  i = getch();
  if (i == 27) errornowait(langtext[LANG_SWABORTLOAD]);
  while (kbhit()) (void)getch();	// flush input buffer (necessary because some keys generate 2 characters)
  printf("\n");
  if (i == 13) return;
#endif

  ensureconsize(lines + 8);

  printf(langtext[LANG_SHOWSWITCHINTRO],
	langtext[LANG_SWTABLEVERCHAR][1], langtext[LANG_SWTABLEVERCHAR][2]);
  putc(' ', stdout);
  for (i=0; i<38; i++)
	putc(langtext[LANG_SWTABLEVERCHAR][3], stdout);
  putc(langtext[LANG_SWTABLEVERCHAR][4], stdout);
  for (i=0; i<38; i++)
	putc(langtext[LANG_SWTABLEVERCHAR][3], stdout);
  printf("\n");

#if DEBUG
{
	// sanity checks
  int k, count;
  for (k=0; k<=lastbit; k++) {
	count = 0;
	for (i=0; i<sizeof(switchorder)/sizeof(switchorder[0]); i++)
		if (switchorder[i] == k)
			count++;
	if ( (k != setsignal1waittime) && (k != setsignal2waittime) &&
	     (k != (WINTTDX?lowmemory:win2k)) && (count != 1) )
		printf("Bit %d is displayed %d times\n", k, count);
  }

}
#endif //  DEBUG

  for (i=0; i<lines; i++) {
	showoneswitch(switchorder[i], 1);
	if (switchorder[i + lines] != -1)
		showoneswitch(switchorder[i + lines], 2);
	else
		printf("\n");
  }

  i = getf(setsignal1waittime);
  if (i || getf(setsignal2waittime)) setf1(setsignal1waittime);
  showoneswitch(setsignal1waittime, 0);
  setf(setsignal1waittime, i);
  if (getf(setsignal1waittime))
	showwaittime(langtext[LANG_SWONEWAY], flags->data.signalwaittimes[0]);
  if (getf(setsignal2waittime))
	showwaittime(langtext[LANG_SWTWOWAY], flags->data.signalwaittimes[1]);

  printf("\n\n ");
  printf(langtext[LANG_SWSHOWLOAD], ttdoptions);

  fflush(stdout);
  if (getch() == 27)
	errornowait(langtext[LANG_SWABORTLOAD]);
  printf("\n");
}

// Dump all switches to swtchlst.txt for parsing by configuration tools
// (in current language too)
int dumpswitches()
{
  FILE *f;
  int i, ch;

  f = fopen("swtchlst.txt", "wt");
  if (!f)
	return 1;

  fprintf(f, "%s\n", TTDPATCHVERSION);
  for (i=0; switches[i].cmdline; i++) {
	if (!switches[i].comment) continue;

	// Write "-[<cmdsw>] <cfgline> <rangemin rangemax|- -> <comment>"
	ch = switches[i].cmdline;
	if ( ((ch & 0xff) > 32) && ((ch & 0xff) < 128) )
		fprintf(f, "-%s ", dchartostr(ch));
	else
		fputs("- ", f);
	fputs(switches[i].cfgcmd, f);
	if ( (switches[i].range[0] != -1) ||
	     (switches[i].range[1] != -1) ) 	// ranged (value)
		fprintf(f, " %ld %ld %ld ",
			switches[i].range[0], switches[i].range[1], switches[i].range[2]);
	else
		fputs(" - - - ", f);
	fprintf(f, langcfg(switches[i].comment), switches[i].cfgcmd, dchartostr(switches[i].cmdline));
	fputs("\n", f);
  }
  fclose(f);
  return 0;
}


// Special debug switches.
// To access, have the first command line argument begin with a "-!";
// the rest of this argument is to have the form
// [<letter>[+|-]]...
// where + sets the feature to always on, - to always off; default is +
// example: ttdpatch -!v-s+ -v
// means ignore version info, always swap, verbose display (-v is processed as usual)
// another example: ttdpatch -!s-c c:/command.com /e:2048
// means run c:/command.com (with 2048 bytes for environment) instead of ttdload.ovl

static const struct debug_switch_struct {
	char c;
	signed char *flag;
} debug_switches[] = {
	{ 'v', &debug_flags.useversioninfo },	// v- ignore version info, v+ collect version info
	{ 's', &debug_flags.swap },		// s+ always swap, s- never swap (DOS only)
	{ 'c', &debug_flags.runcmdline },	// treat the rest of the command line as name+args of program to run
	{ 't', &debug_flags.checkttd },		// t- don't process TTDLOAD[W].OVL
	{ 'm', &debug_flags.checkmem },		// m- run even if memory is too low (DOS only)
	{ 'f', &debug_flags.readcfg },		// f- don't read or create the default ttdpatch.cfg
	{ 'a', &debug_flags.chkswitchdep },	// a- don't check dependencies between switches
	{ 'w', &debug_flags.warnwait },		// w- never wait for key, don't abort; w+ never wait for key, do abort
	{ 'o', &debug_flags.switchorder },	// o+ reorder switches when writing cfg file
	{ 'T', &debug_flags.terminatettd },	// T+ terminate TTD after finding new version info
	{ 'L', &debug_flags.langdatafile },	// L+ load language data from language.dat file
	{ 'I', &debug_flags.noshowleds },	// I- don't update keyboard LED indicators (DOS only)
	{ 'S', &debug_flags.dumpswitches },	// S+ dump all switches to swtchlst.txt and abort
	{ 'C', &debug_flags.protcodefile },	// C+ load protected mode code from ttdprot?.bin file
	{ 0, NULL }
};

void check_debug_switches(int *const argc, const char *const **const argv)
{
  if ((*argc >= 2) && (*argv)[1][0] == '-' && (*argv)[1][1] == '!') {
	const char *sw = (*argv)[1] + 2;
	(*argc)--;
	(*argv)++;

	while (*sw) {
		const struct debug_switch_struct *swdesc;
		for (swdesc = debug_switches; swdesc->c; swdesc++) if (swdesc->c == *sw) {
			signed char val = 1;
			switch (sw[1]) {
				case '-': val = -1;
					// FALLTHROUGH
				case '+': sw++;
			}
			*swdesc->flag = val;
			break;
		}
		sw++;
	}

	if (debug_flags.runcmdline > 0) {
		(*argc)--;
		(*argv)++;
	}
  }
}
