//
// This file is part of TTDPatch
// Copyright (C) 1999, 2000 by Josef Drexler
//
// C++ to C conversion by Marcin Grzegorczyk
//
// windows.c: routines to run the windows-ttd from ttdpatch
//


#include <string.h>
#include <windows.h>

#define IS_WINDOWS_CPP
#include "osfunc.h"
#include "types.h"
#include "error.h"
#include "checkexe.h"
#include "switches.h"
#include "versions.h"
#include "auxfiles.h"

const s32 filesizebase = 1690000;
const s32 filesizeshr = 8;

extern char *patchedfilename;

extern pversioninfo versions[];

const char *ttd_winexenames[] = {
	"GAMEGFX.EXE",
	NULL };

void checkpatch(void)
{
  u32 flen, sections, optheaderstart, sectstart, newentry, entry;
  u32 ourcode, codelen;
  FILE *f, *prcode;

  u32 newexepos = checkexe(&f, ttd_winexenames, maketwochars('P','E'), "Windows");
  u32 section;
  u32 sectionaddr = 0;
  u32 sectionsize;
  u32 sectionalign;
  char sectname[9], *oldcode, *newcode;

  // Ensure that enough memory is available for the large vehicle array
  setseglen(newexepos + 0x50, 0x16, 0x20, 0x80, 0x80);

  // Set section number to 10, and define new section if necessary
  optheaderstart = newexepos + 0x18;
  sectstart = optheaderstart + getval(newexepos + 0x14, 2);	// is ... + optheadersize
  sections = ensureval(newexepos + 6, 2, 10);
  // we have to make the .text section writable so that we can patch the int21handler functions
  ensureval(sectstart + 0x24, 4, getval(sectstart + 0x24, 4) | 0x80000000);
  if (sections) {
	if (sections != 9) {
		fflush(stdout);
		fprintf(stderr, langtext[LANG_TTDLOADINVALID]);
		error(langtext[LANG_DELETEOVL], patchedfilename);
	}
	sectionalign = getval(optheaderstart + 0x20, 4);
	// scan existing sections and find where they end
	for (section = 0; section < sections; section++) {
		// get section's size, round up to next alignment and add to section start -> check if this number is maximum
		u32 sectionend = ((getval(sectstart + 8 + section * 0x28, 4) + sectionalign - 1) & ~(sectionalign - 1)) + getval(0, 4);
		if (sectionend > sectionaddr) sectionaddr = sectionend;
	}
	// patch section fills up to the file's memory image (set in setseglen above)
	sectionsize = getval(newexepos + 0x50, 4) - sectionaddr;
	startwrite();
	fseek(f, sectstart + sections * 0x28, SEEK_SET);
	fwrite("TTDPatch", 8, 1, f);	// section name
	setval(0, 4, sectionsize);		// section virtual size 5MB
	setval(0, 4, sectionaddr);		// virtual address, base 0
	setval(0, 4, 0);		// data in .exe file
	setval(0, 4, 0);		// offset in .exe file
	setval(0, 4, 0);		// offset to relocations
	setval(0, 4, 0);		// offset to line numbers
	setval(0, 2, 0);		// number of relocations
	setval(0, 2, 0);		// number of line numbers
	setval(0, 4, 0xc0000080);	// uninitialised, RW access
  }

  // Make sure our code loader is present
  fseek(f, sectstart + 6 * 0x28, SEEK_SET);
  fread(sectname, 8, 1, f);
  sectname[8] = 0;
  if (strcmp(sectname, "CODESEG")) {
	fflush(stdout);
	fprintf(stderr, langtext[LANG_TTDLOADINVALID]);
	error(langtext[LANG_DELETEOVL], patchedfilename);
  }
  ourcode = getval(0, 4);		// virtual size
  newentry = ourcode + getval(0, 4);	// + address = end of CODESEG
  getval(0, 4);	// skip 4 bytes
  ourcode += getval(0, 4);		// + .exe offset = same location in .exe file

#ifdef USEPROTBIN
  if (!findattachment(AUX_PROTCODE, &flen, &prcode))
	error(langtext[LANG_INTERNALERROR], 7);

  fseek(prcode, flen, SEEK_SET);
  fread(&codelen, 4, 1, prcode);

  newcode = malloc(codelen);
  fread(newcode, 1, codelen, prcode);
#else
  codelen = (u32) &winloaderend- (u32) &winloader;
  newcode = (char*) &winloader;
#endif
  oldcode = malloc(codelen);

  // make sure the new loader is valid
  if ( *(u32*) (newcode+2) != MAGIC )
	error(langtext[LANG_INTERNALERROR], 5);

  fseek(f, ourcode, SEEK_SET);
  fread(oldcode, 1, codelen, f);
  entry = *(u32*) (oldcode+2);			// get real entry point of current code loader
  *(u32*) (newcode+2) = entry;			// so that comparing won't fail b/o this
  if (memcmp(oldcode, newcode, codelen)) {		// empty code
	printf(langtext[LANG_INSTALLLOADER]);
	startwrite();
	fseek(f, ourcode, SEEK_SET);
	fwrite(newcode, 1, codelen, f);

	if (entry)
		setval(ourcode + 2, 4, entry);		// keep original entry point

		// Set entry point to our code loader
	entry = ensureval(optheaderstart + 0x10, 4, newentry);
	if (entry)
		if (!getval(ourcode + 2, 4))
			setval(ourcode + 2, 4, entry + 0x400000);
  }

#ifdef USEPROTBIN
  free(newcode);
#endif

  fseek(f, 0, SEEK_END);
  flen = ftell(f);

  fclose(f);

  printf(langtext[LANG_TTDLOADOK], patchedfilename);

  checkversion(newversion, flen);
  loadingamestrings(flen);
}


// fix the HDPath in the registry, to not end in NULL
// and to be a 8.3 pathname and to end with a backslash
void fixregistry(void)
{
  HKEY hkey;
  int modified;

  char *filename, *shortname;
  DWORD len, shortlen;
  DWORD type, result;

  // open the key
  result = RegOpenKey(HKEY_LOCAL_MACHINE,
	"Software\\Fish Technology Group\\Transport Tycoon Deluxe",
	&hkey);

  if (result)
	error(langtext[LANG_REGISTRYERROR], 1);

  // get the length of the path
  result = RegQueryValueEx(hkey, "HDPath", NULL, &type, NULL, &len);
  if (result)
	error(langtext[LANG_REGISTRYERROR], 2);

  filename = malloc(len);

  // get the value of the installation path
  result = RegQueryValueEx(hkey, "HDPath", NULL, &type, (BYTE*) filename, &len);
  if (result)
	error(langtext[LANG_REGISTRYERROR], 3);

  shortlen = len;
  shortname = malloc(shortlen);

  // translate to a short path name
  result = GetShortPathName(filename, shortname, shortlen);
  if (!result)
	error(langtext[LANG_REGISTRYERROR], 4);

  if (result != strlen(shortname)) {
	// buffer too small (how did that happen??)

	shortlen = result;
	realloc(shortname, shortlen);
	result = GetShortPathName(filename, shortname, shortlen);
	if (!result || (result != strlen(shortname)))
		error(langtext[LANG_REGISTRYERROR], 5);
  }

  modified = 0;
  if (stricmp(filename, shortname)) {
	// Not the same, so it was a long name.  Save the short name.
	modified |= 1;
  }
  if (strlen(filename)+1 != len) {
	// Same, but data was too long.  Probably had a NULL, so
	// simply save it back and get rid of the NULL.

	modified |= 2;
  }
  if (shortname[strlen(shortname)-1] != '\\') {
	// path doesn't end in a backslash.

	if (shortlen < strlen(shortname)+2) {
		shortlen++;
		shortname = realloc(shortname, shortlen);
	}
	strcat(shortname, "\\");
	modified |= 4;
  }

  if (modified) {
	result = RegSetValueEx(hkey, "HDPath", 0, type,
			(BYTE*) shortname, strlen(shortname));
	if (result)
		error(langtext[LANG_REGISTRYERROR], modified | 8);
  }

  // close the key
  RegCloseKey(hkey);

  free(shortname);
  free(filename);

}

void prepare_exec()
{
}

int createchildprocess(char *commandline)
{
   PROCESS_INFORMATION piProcInfo;
   STARTUPINFO siStartInfo;

// Set up members of the PROCESS_INFORMATION structure.

   ZeroMemory(&piProcInfo, sizeof(PROCESS_INFORMATION) );

// Set up members of the STARTUPINFO structure.

   ZeroMemory(&siStartInfo, sizeof(STARTUPINFO) );
   siStartInfo.cb = sizeof(STARTUPINFO);

// Create the child process.

   return CreateProcess(NULL,
	commandline,	// command line
	NULL,		// process security attributes
	NULL,		// primary thread security attributes
	TRUE,		// handles are inherited
	0,		// creation flags
	NULL,		// use parent's environment
	NULL,		// use parent's current directory
	&siStartInfo,	// STARTUPINFO pointer
	&piProcInfo);	// receives PROCESS_INFORMATION
}

void makehandlepair(int handletype, SECURITY_ATTRIBUTES *sa,
	HANDLE *saved, HANDLE *child)
{
  HANDLE pipe[2];	// [0] = read, [1] = write
  int success, handleofs;

   // Comments apply to "STDOUT" case. For stdin, switch read/write

  *saved = GetStdHandle(handletype);

   // Create a pipe for the child process's STDOUT.

  if (!CreatePipe(pipe, pipe+1, sa, 0))
	error("Pipe creation failed\n");

   // Set a write handle to the pipe to be STDOUT.

  handleofs = (handletype==STD_OUTPUT_HANDLE);
  if (!SetStdHandle(handletype, pipe[handleofs]))
	error("Redirection failed");

   // Create noninheritable read handle and close the inheritable read
   // handle.

  success = DuplicateHandle(GetCurrentProcess(), pipe[1-handleofs],
	GetCurrentProcess(), child, 0, FALSE, DUPLICATE_SAME_ACCESS);

  if (!success)
	error(langtext[LANG_RUNERROR], "", "DuplicateHandle failed");

  CloseHandle(&(pipe[1-handleofs]));
}


int writepatchdata(HANDLE dat)
{
  int success;
  unsigned long patchdatsize, patchmemsize, written, size[2];

#ifdef USEPROTBIN
  u32 ofs;
  FILE *f;
  char *data;

  if (!findattachment(AUX_PROTCODE, &ofs, &f))
	error(langtext[LANG_INTERNALERROR], 8);

  fseek(f, ofs, SEEK_SET);
  fread(&patchmemsize, 4, 1, f);	// loader size
  fseek(f, patchmemsize, SEEK_CUR);

  fread(&patchmemsize, 4, 1, f);
  fread(&patchdatsize, 4, 1, f);
  fseek(f, sizeof(paramset), SEEK_CUR);	// skip flags

  data = malloc(patchdatsize+8);

  memcpy(data, &patchmemsize, 4);
  memcpy(data+4, &patchdatsize, 4);
  memcpy(data+8, flags, sizeof(paramset));

  written = 8+sizeof(paramset);
  patchdatsize += 8;

  fread(data+written, 1, patchdatsize-written, f);

#else
  _protptr data;

  data = (_protptr) &protectedcode;

	// second dword in protectedcode is initialized size to write
  patchdatsize = *(((u32 *) data) + 1) + 8;
#endif

  success = WriteFile(dat, data, patchdatsize, &written, NULL);
  if (!success || (written != patchdatsize))
	return 0;

  if (curversion->h.numoffsets) {
	size[0] = size[1] = sizeof(versionheader) + 4 * curversion->h.numoffsets;
  } else {
	// no version info, still reserve space for numoffsets
	curversion->h.numoffsets = ALLOCEMPTYOFFSETS;
	size[1] = sizeof(versionheader);
	size[0] = size[1] + 4 * curversion->h.numoffsets;
  }

  if (!WriteFile(dat, size, 8, &written, NULL)) return 0;
  if (!WriteFile(dat, curversion, size[1], &written, NULL)) return 0;

  size[0] = size[1] = customtextsize;
  if (!WriteFile(dat, size, 8, &written, NULL)) return 0;
  if (!WriteFile(dat, customtexts, size[1], &written, NULL)) return 0;

  return 1;
}


#ifdef __BORLANDC__
#pragma argsused
#endif
int runttd(const char *program, char *options, langinfo **info)
{
  HANDLE childstdin, childstdout, savestdin, savestdout;

  SECURITY_ATTRIBUTES sa;
  int cmdlength, success;
  char *commandline;
  char *ptr;
  unsigned long remain;
#ifdef USEPATCHDAT
  char *filename = "patchdat.dat";
#endif


  fixregistry();

  cmdlength = strlen(options) + strlen(program) + 1;
  commandline = (char*) malloc(cmdlength);
  if (!commandline)
	error(langtext[LANG_NOTENOUGHMEM], "runttd_win()", cmdlength/1024+1);

  strcpy(commandline, program);
  if (strlen(options)) {
	strcat(commandline, " ");
	strcat(commandline, options);
  }

//  memmove(options + strlen(program) + 1, options, sizeof(options) - strlen(program) - 1);
//  *stpcpy(options, program) = ' ';

  printf(langtext[LANG_RUNTTDLOAD], commandline, "", "");

   // Set the bInheritHandle flag so pipe handles are inherited.

  sa.nLength = sizeof(SECURITY_ATTRIBUTES);
  sa.bInheritHandle = TRUE;
  sa.lpSecurityDescriptor = NULL;

   // Create handle pairs for stdin and stdout
  makehandlepair(STD_OUTPUT_HANDLE, &sa, &savestdout, &childstdout);
  makehandlepair(STD_INPUT_HANDLE, &sa, &savestdin, &childstdin);

   // Now create the child process.
  if (!createchildprocess(commandline)) {
	printf (langtext[LANG_RUNERROR], program, "Create process failed");
	error(langtext[LANG_DELETEOVL], patchedfilename);
  }

  free(commandline);

   // After process creation, restore the saved STDIN and STDOUT.
  if (!SetStdHandle(STD_INPUT_HANDLE, savestdin))
	error(langtext[LANG_RUNERROR], program, "Re-redirecting Stdin failed\n");
  if (!SetStdHandle(STD_OUTPUT_HANDLE, savestdout))
	error(langtext[LANG_RUNERROR], program, "Re-redirecting Stdout failed\n");

   // Write to pipe that is the standard input for a child process.
#ifdef USEPATCHDAT
  {
	long length = (long) &protectedfuncrealend - (long) &protectedfunc;
	FILE *patchdata = fopen(filename,"wb");
	fwrite(&protectedfunc, length, 1, patchdata);
	fclose(patchdata);
  }
#else
  if (!writepatchdata(childstdin))
	error(langtext[LANG_RUNERROR], program, "Writing to the pipe\n");
#endif

   // Read the return information
  if (!getf(recordversiondata))
	return 0;		// no version data if not recording version data (Duh!)

  {
	versionheader hdr;
	unsigned long read;

	success = ReadFile(childstdout, &hdr, sizeof(versionheader), &read, NULL);
	if (!success || (read != sizeof(versionheader)) )
		success = 0;

	curversion = malloc(sizeof(versionheader) + hdr.numoffsets * 4);
	curversion->h = hdr;

	ptr = (char*) &curversion->versionoffsets;
	remain = 4*hdr.numoffsets;
	while (remain && success) {
		success = ReadFile(childstdout, ptr, remain, &read, NULL);

		ptr += read;
		remain -= read;
	}

#if defined(USEPATCHDAT) && !DEBUG
	remove(filename);
#endif

	if (!success || (curversion->h.version != MOREMAGIC) )
		return 1;
  }
  return 0;
}

s16 getdefaultlanguage(langinfo *info)
{
  char langid[6];
  if (!GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_ILANGUAGE, langid, sizeof(langid)))
	error("Could not get Windows locale information.\n");

  {
    int numlangid = strtol(langid, NULL, 16);

    return langinfo_findlangfromlid(info, numlangid & 0xff);
  }
}

void ensureconsize(int minlines)
{
  CONSOLE_SCREEN_BUFFER_INFO info;
  HANDLE handle = CreateFile("CONOUT$",
			     GENERIC_READ | GENERIC_WRITE,
			     FILE_SHARE_READ | FILE_SHARE_WRITE,
			     NULL,
			     OPEN_EXISTING,
			     0,
			     0);
  if (handle != INVALID_HANDLE_VALUE && GetConsoleScreenBufferInfo(handle, &info)) {
    if (info.dwSize.X < 80 || info.dwSize.Y < minlines) {
      info.srWindow.Left = 0;
      info.srWindow.Top = 0;
      info.srWindow.Right = 79;
      info.srWindow.Bottom = 49;
      SetConsoleWindowInfo(handle, 1, &info.srWindow);
      info.dwSize.X = 80;
      info.dwSize.Y = 50;
      SetConsoleScreenBufferSize(handle, info.dwSize);
    }
  }
}

void restoreconsize()
{
	// this doesn't work yet
}

/*
void addgrffiles(FILE *f)
{
  int done;
  WIN32_FIND_DATA fblk;
  HANDLE fhnd;
  char *fname = (char*) &fblk.cFileName;

  // look for *.adw files in the newgrf directory
  // add the corresponding .grf files to newgrfw.txt and
  // delete the .adw file

  SetCurrentDirectory("newgrf");
  fhnd = FindFirstFile("*.adw", &fblk);
  done = (fhnd == INVALID_HANDLE_VALUE);

  while (!done) {
	fprintf(f, "newgrf/%.*s.grf\n", (int) strlen(fname) - 4, fname);
	remove(fname);

	done = !FindNextFile(fhnd, &fblk);
  }

  FindClose(fhnd);
  SetCurrentDirectory("..");
}
*/

#define MAX_SAVE_CONSOLE_TITLE	1024

void iconproc(UINT msg, HICON bigin, HICON smallin, HICON *bigout, HICON *smallout)
{
  HICON smallico, bigico;
  static HWND hConsole;
  static BOOL initialized;

  if (!initialized) {
	static char titlebuf[MAX_SAVE_CONSOLE_TITLE];
	const char *temptitle = "TTDPatch: PID=%X";
	static char temptitlebuf[32];

	initialized = 1;

	// No API to get handle of a process's console window exists before Win2000,
	// so we resort to an Official Microsoft Kludge.

	// save the current console window's title
	if (!GetConsoleTitleA(titlebuf, MAX_SAVE_CONSOLE_TITLE)) return;

	// set console window's title to a temporary unique string
	sprintf(temptitlebuf, temptitle, (unsigned)GetCurrentProcessId());
	if (!SetConsoleTitleA(temptitlebuf)) return;

	Sleep(40);	// as in the example on the MS support site...

	// now find that window via a documented API
	hConsole = FindWindowA(NULL, temptitlebuf);	// lpClassName=NULL matches all classes
							// (documented in the MSDN Library)

	// restore the original title
	SetConsoleTitleA(titlebuf);
  }

  if (!hConsole) return;	// failed to obtain the console handle the first time, better not try again

  bigico = (HICON) SendMessage( hConsole, msg, ICON_BIG, (LPARAM) bigin);
  smallico = (HICON) SendMessage( hConsole, msg, ICON_SMALL, (LPARAM) smallin);

  if (bigout) *bigout = bigico;
  if (smallout) *smallout = smallico;
}

HICON oldbigicon, oldsmallicon;

void restoreicon(void)
{
  iconproc(WM_SETICON, oldbigicon, oldsmallicon, 0, 0);
}

void initializewindow(void)
{
  // set icon for console to first icon from the .exe resources

  HICON ico = LoadIcon( GetModuleHandle( NULL ), MAKEINTRESOURCE( 1 ) );

  if ( ico != NULL ) {
	iconproc(WM_GETICON, 0, 0, &oldbigicon, &oldsmallicon);
	iconproc(WM_SETICON, ico, ico, 0, 0);
	atexit(restoreicon);
  }
}
