/*
 * MCI to DirectMusic wrapper
 *
 * This file contains the DirectMusic calls.
 *
 */



#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

// for gcc, the GUIDs are available in a library instead
#ifndef __GNUC__
#define INITGUID
#endif

#include <windows.h>
#include <stdio.h>

#define VARIANT int

#include <dmksctrl.h>
#include <dmusici.h>
#include <dmusicc.h>
#include <dmusicf.h>

#define MSGBOX(output)  MessageBox(NULL,output,"dxmci",MB_OK);
#define MULTI_TO_WIDE( x,y )  MultiByteToWideChar( CP_ACP,MB_PRECOMPOSED, y,-1,x,_MAX_PATH);

// the performance object controls manipulation of  the segments
IDirectMusicPerformance *performance = NULL;

// the segment object is where the MIDI data is stored for playback
IDirectMusicSegment *segment = NULL;

// the loader bject can load many types of DMusic related files
IDirectMusicLoader *loader = NULL;						

// whether we've initialized COM or not (when deciding whether to shut down)
int COMInitialized = 0;

// Initialize COM and DirectMusic
bool InitDirectMusic (void)
{
	if (NULL != performance)
		return true;

	// Initialize COM
	if (!COMInitialized) {
		CoInitialize (NULL);
		COMInitialized = 1;
	}

	// Create the performance object via CoCreateInstance
	if (FAILED(CoCreateInstance(
			(REFCLSID)CLSID_DirectMusicPerformance,
			NULL,
			CLSCTX_INPROC,
			(REFIID)IID_IDirectMusicPerformance,
			(LPVOID *)&performance))) {
		MSGBOX ("Failed to create the performance object");
		return false;
	}

	// Initialize it
	if (FAILED(performance->Init(NULL, NULL, NULL))) {
		MSGBOX("Failed to initialize performance object");
		return false;
	}

	// Choose default Windows synth
	if (FAILED(performance->AddPort (NULL))) {
		MSGBOX("AddPort failed");
		return false;
	}

	// now we'll create the loader object. This will be used to load the
	// midi file for our demo. Again, we need to use CoCreateInstance
	// and pass the appropriate ID parameters
	if (FAILED(CoCreateInstance ((REFCLSID)CLSID_DirectMusicLoader,
			NULL, CLSCTX_INPROC, 
			(REFIID)IID_IDirectMusicLoader,
			(LPVOID *)&loader))) {
		MSGBOX("Failed to create loader object");
		return false;
	}

	// that's it for initialization. If we made it this far we
	// were successful.
	return true;
}

// Releases memory used by all of the initialized 
// DirectMusic objects in the program
void ReleaseSegment (void)
{
	if (NULL != segment) {
		segment->Release ();
		segment = NULL;
	}
}
void ShutdownDirectMusic (void)
{
	// release everything but the segment, which the performance 
	// will release automatically (and it'll crash if it's been 
	// released already)

	if (NULL != loader) {
		loader->Release ();
		loader = NULL;
	}

	if (NULL != performance)
	{
 		performance->CloseDown ();
		performance->Release ();
		performance = NULL;
	}

	if (COMInitialized) {
		CoUninitialize ();
		COMInitialized = 0;
	}
}

// Load MIDI file for playing 
bool LoadMIDI (char *directory, char *filename)
{
	DMUS_OBJECTDESC obj_desc;
	WCHAR w_directory[_MAX_PATH];	// utf-16 version of the directory name.
	WCHAR w_filename[_MAX_PATH];	// utf-16 version of the file name

	if (NULL == performance)
		return false;

	MULTI_TO_WIDE(w_directory,directory);

	if (FAILED(loader->SetSearchDirectory((REFGUID) GUID_DirectMusicAllTypes,
				w_directory, FALSE))) {
		MSGBOX("LoadMIDI: SetSearchDirectory failed");
		return false;
	}

	// set up the loader object info
	ZeroMemory (&obj_desc, sizeof (obj_desc));
	obj_desc.dwSize = sizeof (obj_desc);

	MULTI_TO_WIDE(w_filename,filename);
	obj_desc.guidClass = CLSID_DirectMusicSegment;

	wcscpy (obj_desc.wszFileName,w_filename);
	obj_desc.dwValidData = DMUS_OBJ_CLASS | DMUS_OBJ_FILENAME;

	// release the existing segment if we have any
	if (NULL != segment)
		ReleaseSegment();

	// and make a new segment
	if (FAILED(loader->GetObject(&obj_desc, 
			(REFIID)IID_IDirectMusicSegment, 
			(LPVOID *) &segment))) {
		MSGBOX("LoadMIDI: Get object failed");
		return FALSE;
	}

	// next we need to tell the segment what kind of data it contains. We do this
	// with the IDirectMusicSegment::SetParam function.
	if (FAILED(segment->SetParam((REFGUID)GUID_StandardMIDIFile,
			-1, 0, 0, (LPVOID)performance))) {
		MSGBOX("LoadMIDI: SetParam (MIDI file) failed");
		return false;
	}

	// finally, we need to tell the segment to 'download' the instruments
	if (FAILED(segment->SetParam((REFGUID)GUID_Download,
			-1, 0, 0, (LPVOID)performance))) {
		MSGBOX("LoadMIDI: Failed to download instruments");
		return false;
	}

	// at this point, the MIDI file is loaded and ready to play!
	return true;
}

// Start playing the MIDI file
void PlaySegment (void)
{
	if (NULL == performance)
		return;

	if (FAILED(performance->PlaySegment(segment, 0, 0, NULL))) {
		MSGBOX("PlaySegment failed");
	}
}

// Stop playing
void StopSegment (void)
{
	if (NULL == performance || NULL == segment)
		return;

	if (FAILED(performance->Stop(segment, NULL, 0, 0))) {
		MSGBOX("StopSegment failed");
	}
}

// Find out whether playing has started or stopped
bool IsSegmentPlaying (void)
{
	if (NULL == performance || NULL == segment)
		return FALSE;

	// IsPlaying return S_OK if the segment is currently playing
	return performance->IsPlaying(segment, NULL) == S_OK ? TRUE : FALSE;
}


#if defined(__GNUC__)
#define ALIGN_ESP   asm("movl %%esp,%0" : "=m" (savedesp) ); \
		    asm("andl $-3,%esp" );
#define RESTORE_ESP asm("movl %0,%%esp" : : "m" (savedesp) );
#else
#define ALIGN_ESP   _asm mov savedesp, esp; \
		    _asm and esp, not 3;
#define RESTORE_ESP _asm mov esp, savedesp;
#endif

#if defined(__cplusplus)
extern "C"
{
#endif


__declspec(dllexport) BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fwdReason, LPVOID lpvReserved)
{

	switch (fwdReason) {
//	case DLL_PROCESS_ATTACH:
//		asm("int3");
//		//MSGBOX("Process Attach");
//		break;
	case DLL_PROCESS_DETACH:
//		asm("int3");
		if (COMInitialized)
			ShutdownDirectMusic();
		//MSGBOX("Process Detach");
		break;
	}
	return TRUE;
}

__declspec(dllexport) MMRESULT dxMidiGetVolume(HMIDIOUT hmo, LPDWORD pdwVolume)
{
	DWORD savedesp;

	if (NULL == performance) {
		ALIGN_ESP
		InitDirectMusic();
		RESTORE_ESP
	}

	return 0;
}

__declspec(dllexport) MMRESULT dxMidiSetVolume(HMIDIOUT hmo, DWORD dwVolume)
{
	DWORD savedesp;
	long db;

	ALIGN_ESP

	if (NULL == performance)
		InitDirectMusic();

	db = ((dwVolume >> 21) & 0x7ff) - 1000;
	performance->SetGlobalParam(GUID_PerfMasterVolume, &db, sizeof(db));

	RESTORE_ESP

	return 0;
}

#define STAT_SEEKING 0
#define STAT_PLAYING 1
#define STAT_STOPPED 2
char *status[] = { "seeking", "playing", "stopped" };

__declspec(dllexport) MCIERROR dxMidiSendString(LPCTSTR lpstrCommand, LPTSTR lpstrReturnString, UINT uReturnLength, HWND hwndCallback)
{
	DWORD savedesp;
	char command, *pos;
	char dir[_MAX_PATH+1];
	char file[_MAX_PATH+1];

	static bool seeking;

	ALIGN_ESP

	command = lpstrCommand[0];
	if (NULL == performance) {
		// initialize DirectMusic, unless this is either
		// a "close all" or a "status" command.  Skip the "close all".
		if (command == 'c')
			command = 'i';	// for "ignore" :)
		else if (command != 's')
			InitDirectMusic();
	}

	// check only first character, it's unique
	switch(command) {
		case 'o':	// "open <fullpath> type sequencer alias MUSIC"

			// split full path into directory and file components
			strncpy(dir, lpstrCommand + 5, _MAX_PATH);
			pos = strrchr(dir, '\\') + 1;
			strncpy(file, pos, _MAX_PATH);
			strchr(file, ' ')[0] = 0;
			*pos = 0;
			LoadMIDI(dir, file);
			break;
		case 'p':	// "play MUSIC from 0"
			PlaySegment();
			seeking = 1;
			break;
		case 's':	// "status MUSIC mode"

			// if a segment has been defined, and it was "seeking",
			// check whether it's done seeking yet
			if (segment && seeking && IsSegmentPlaying())
				seeking = false;

			// then return the appropriate result
			strcpy(lpstrReturnString, status
				[
				  !segment ?		STAT_STOPPED :
				  seeking ?		STAT_SEEKING :
				  IsSegmentPlaying() ?	STAT_PLAYING :
							STAT_STOPPED
				] );
			break;
		case 'c':	// "close all"
			StopSegment ();
			break;
	}

	RESTORE_ESP

	return 0;
}

#if defined(__cplusplus)
}
#endif 
