
// Various tools needed
// e.g. ctrl key checking

#define CTRL_DEFAULT 0	// only human player, and only in single player
#define CTRL_ANY 1	// any player can be active (use only if current player is undefined)
#define CTRL_MP 2	// multiplayer games too

// also all of the following are valid for ctrlkeystate:

#define PL_DEFAULT 0	// only real human player
#define PL_PLAYER 4	// assume current player, stored in high byte of low word on stack
#define PL_NOTTEMP 8	// but not ai companies temporarily managed by human
#define PL_ORG 16	// also original human companies temporarily unmanaged
#define PL_RETURNCURRENT 32	// return current player in ecx

	// checks if a ctrl key is pressed
	// in:	on stack, above CTRL bit(s)
	// out: zf is pressed, nz if not
ctrlkeystate:
	push ecx

	test byte [esp+8],CTRL_ANY
	jnz short .ishuman	// any player is ok

	push dword [esp+8]
	call ishumanplayer
	jnz short .haveit	// computer players never press their Ctrl key

.ishuman:
	test byte [esp+8],CTRL_MP
	jnz short .issingleplayer

	// in multiplayer, Ctrl is disabled
	cmp byte [numplayers],1
	jne short .haveit

.issingleplayer:
	mov ecx,keypresstable+0x80	// make offsets fit in a byte

#if WINTTDX
	cmp byte [ecx-0x80+KEY_Ctrl],0
#else
	cmp byte [ecx-0x80+KEY_LCtrl],0
	jz short .haveit
	cmp byte [ecx-0x80+KEY_RCtrl],0
#endif
.haveit:
	pop ecx
	ret 4
; endp ctrlkeystate


	// checks if the current player is a human player
	// in:	PL_xxx flags on stack
	// out:	ecx with the current player if PL_RETURNCURRENT is set
	// 	nz if not a human player
	//	zf if a human player.  In that case, we also
	//	clear carry if first human player, set carry if second one
proc ishumanplayer
	arg playertype
	local player

	_enter

	push eax
	push ecx

	movzx eax,byte [curplayer]

	test byte [%$playertype],PL_PLAYER
	jz short .notspecific

	mov al,[%$playertype+1]

.notspecific:
	mov ecx,[human1]

	test byte [%$playertype],PL_NOTTEMP
	jz short .curhuman

	// humans temporarily managing an AI company don't count
	// i.e. if cl!=human1 or ch!=human2, remove from the list

	cmp cl,[landscape3+ttdpatchdata.orgpl1]
	je short .nottemp1

	mov cl,0xf0

.nottemp1:
	cmp ch,[landscape3+ttdpatchdata.orgpl2]
	je short .nottemp2

	mov ch,0xf0

.nottemp2:

.curhuman:
	mov [%$player],ecx

	mov cx,0xf0f0
	test byte [%$playertype],PL_ORG
	jz short .notorg

	// also wants original human companies
	mov ecx,[landscape3+ttdpatchdata.orgpl1]
	and cx,0x7f7f

.notorg:
	cmp al,cl
	je short .knowit
	cmp al,byte [%$player]
	je short .knowit

	cmp al,ch
	je short .player2

	cmp al,[%$player+1]

.player2:
	stc

.knowit:
	pop ecx
	lahf
	test byte [%$playertype],PL_RETURNCURRENT
	jz short .notreturn

	movzx ecx,al

.notreturn:
	sahf
	pop eax
	_ret
endproc // ishumanplayer

// check only real human player
// this is to save 2 bytes (PUSH BYTE PL_DEFAULT) per each instance of the most common check
isrealhumanplayer:
	push byte PL_DEFAULT
	call ishumanplayer
	ret


// handle memory allocation at the end of the heap

// in:	on stack: requested size
// out:	on stack: pointer (will be DWORD aligned)
//	carry set and pointer is -1 if out of memory
spritemalloc:
heapmalloc:
	pusha

	mov eax,[esp+0x24]
	add eax,byte 3
	and eax,byte ~3

	mov esi,[heapptr]

.trynext:
	cmp [esi+heap.left],eax
	jae .gotit

	mov esi,[esi+heap.next]
	test esi,esi
	jnz .trynext
	jmp short .allocmore

.gotit:
	sub [esi+heap.left],eax
	xchg eax,[esi+heap.ptr]
	add [esi+heap.ptr],eax

.return:
	mov [esp+0x24],eax

	popa
	ret

.allocmore:
#if !WINTTDX
	// try making DS larger

// XXX: Will this work for LINTTDX ?! We usually patch these as we can't
// allocate you well real-mode memory. --pasky
// Actually, this shouldn't ever happen as we should have enough heap reserved
// on the start already. So we will leave this as-is so that we can catch the
// case where we wouldn't. --pasky

	push eax
	push es
	mov ebx,ds
	mov es,ebx
	lsl ebx,ebx
	inc ebx
	shr ebx,12
	add ebx,32		// request memory in 32 4K page chunks = 128KB
	mov ah,0x4a
	int 0x21		// re-size allocated memory
	pop es
	jc .fail

.ok:
	pop eax
	mov esi,[varheap]
	mov ebx,ds
	lsl ebx,ebx
	inc ebx
	sub ebx,[esi+heap.ptr]
	mov [esi+heap.left],ebx
	jmp .trynext

#else

uvard varheapsize	// current varheap chunk size in bytes

	push eax
	mov ecx,2		// try reserving not more than twice
	mov esi,[varheap]
	test esi,esi
	jz .reserve

.commit:
	// commit in 128 KB chunks

	mov eax,[varheapsize]
	add eax,32*4096
	mov ebx,eax
	push ecx

	push 4			// PAGE_READWRITE
	push 0x1000		// AllocateType MEM_COMMIT
	push eax		// dwSize
	push esi		// Address
	call dword [0x423424]	// VirtualAlloc()
	pop ecx
	test eax,eax
	jz .reserve

	cmp dword [esi+heap.ptr],esi
	ja .notnew

	lea eax,[esi+heap_size]
	mov [esi+heap.ptr],eax
	or dword [esi+heap.left],byte -heap_size

.notnew:
	mov [varheapsize],ebx
	add dword [esi+heap.left],32*4096
	pop eax
	jmp .trynext

.reserve:

	// find end of heap chain to append this chunk to
	mov eax,[heapptr]

.next:
	xchg eax,esi
	mov eax,[esi+heap.next]
	test eax,eax
	jnz .next

	// now esi=last valid heap structure

	// reserve in 2 MB chunks
	push ecx

	push 4			// PAGE_READWRITE
	push 0x2000		// AllocateType MEM_RESERVE
	push 0x200000		// Reserve (without committing) a 2 MB chunk
	push 0			// Address
	call dword [0x423424]	// VirtualAlloc()
	pop ecx
	test eax,eax
	jz .fail

	mov [esi+heap.next],eax
	mov [varheap],eax
	and dword [varheapsize],0
	xchg eax,esi
	loop .commit		// don't try committing more than twice

#endif

.fail:
	stc
	pop eax
	sbb eax,eax
	jmp .return


// Force the whole screen to be redrawn
redrawscreen:
	pusha
	xor ax,ax
	mov dx,640
	xor bx,bx
	mov bp,480
	call dword [invalidaterect]
	popa
	ret
; endp redrawscreen

#if 0	// currently not used
// Find the first vehicle in a consist (linked via vehicle.nextwaggon)
// in:	EAX->any vehicle in a consist
// out:	ESI->first vehicle
findfirstvehicle:
	push eax
	push edi

.again:
	mov edi,[veharrayptr]
	mov esi,dword [eax+veh.idx]	// no 66h prefix, and we ignore the high word anyway

	// search for the previous vehicle
.vehicleloop:
	cmp byte [edi+veh.class],0
	je short .nextvehicle
	cmp si,word [edi+veh.nextunitidx]
	je short .found

.nextvehicle:
	sub edi,byte -vehiclesize
	cmp edi,[veharrayendptr]
	jb short .vehicleloop

	// no previous vehicle found, this is the first one
	xchg esi,eax	// mov esi,eax in 1 byte, eax is discarded anyway
	pop edi
	pop eax
	ret

.found:
	xchg eax,edi	// mov eax,edi in 1 byte, edi is discarded anyway
	jmp short .again
; endp findfirstvehicle
#endif


// Find the table and offset of a TTD text ID
//
// in:	AX=text id (can be specific or general)
//	doesn't work for text IDs 7800..7FFF (refer to customstringarray for those)
// out:	EAX=table ptr
//	EDI=text ptr
gettextandtableptrs:
	call gettextintableptr
	mov edi,[eax+edi*4]
	jnc .notadded
	add edi,eax
.notadded:
	ret

// same as above, except:
// out:	EDI=offset to text ptr
//	carry set if EAX must be added to text ptr
gettextintableptr:
	mov edi,eax
	and edi,0x7ff
	// fall through to gettexttableptr

// same as above, except it does not return offset to text ptr and does not use EDI
gettexttableptr:
	movzx eax,ah
	and al,-8
	jz .gentable

	cmp al,-8
	je .ttdpatchtable

	sub al,8

	push ebx
	mov ebx,eax
	shr ebx,3
	movzx ebx,byte [texttableoffsets+ebx]

	add eax,[mainhandlertable]
	mov eax,[eax]
	mov eax,[eax+8]
	mov eax,[eax+ebx]
	pop ebx
	// carry is clear here
	ret

.gentable:
	mov eax,[mainstringtable]
	// and here too
	ret

.ttdpatchtable:
	mov eax,[customtextptr]
	stc
	ret
; endp gettexttableptr

var texttableoffsets, db 4,4,4,18,4,18,4,4,4,4,4,4,4,16,-128,4,4,4,4,4,4,4

// finds a vehicle on a given tile
// I don't know which one it returns if there are multiple vehicles on the tile.
// in: di: tile XY coords
// out: edi: pointer to vehicle data or -1 if there are no vehs
// zf set if no vehicles are found
findvehontile:
	pusha
	mov dword [foundveh],-1
	mov [vehcoords],di
	mov eax,addr(findvehontile_helper)
	call dword [searchcollidingvehs]
	popa
	mov edi,[foundveh]
	cmp edi,byte -1
	ret

findvehontile_helper:
	push eax
	mov ax,[vehcoords]
	cmp ax,[esi+veh.XY]
	jnz .exit
	movzx eax, word [esi+veh.engineidx]
	shl eax,vehicleshift
	add eax,dword [veharrayptr]
	mov [foundveh],eax
.exit:
	pop eax
	ret

uvarw vehcoords
uvard foundveh

// Decides whether a schedule is shared (referred by more than one vehicle)
// in: ebp: pointer to schedule
// out: cf set if shared
isscheduleshared:
	push eax
	push ebx
	mov ebx,2
	mov eax,[veharrayptr]
	
.checkvehicle:
	cmp byte [eax+veh.class],0
	je .nextveh
	cmp ebp,[eax+veh.scheduleptr]
	jne .nextveh
	dec ebx
	jz .shared

.nextveh:
	sub eax,byte -vehiclesize
	cmp eax,[veharrayendptr]
	jb .checkvehicle

	clc
	jmp short .end

.shared:
	stc

.end:
	pop ebx
	pop eax
	ret

// the following procs all serve to maintain the veh.engineidx variable

createvehentry:
	push eax
	mov word [esi+veh.nextunitidx],-1
	mov ax,[esi+veh.idx]
	mov [esi+veh.engineidx],ax
	call dword [randomfn]
	mov [esi+veh.random],al
	mov byte [esi+veh.newrandom],0
	pop eax
	ret

// attach new vehicle to previous vehicle of consist
proc attachveh
	arg new,previous

	_enter
	push eax
	push esi

	// old+veh.nextunitidx = new+veh.idx
	// new+veh.engineidx = old+veh.engineidx

	mov esi,[%$new]
	mov ax,[esi+veh.idx]

	mov esi,[%$previous]
	mov [esi+veh.nextunitidx],ax

	mov ax,[esi+veh.engineidx]

	mov esi,[%$new]
	mov [esi+veh.engineidx],ax

	pop esi
	pop eax
	_ret
endproc

// remove vehicle from consist
proc detachveh
	arg removed,previous

	_enter
	push eax
	push esi

	// previous.nextunitidx = removed.nextunitidx
	// removed.nextunitidx will be cleared/set later

	mov esi,[%$removed]

	mov ax,[esi+veh.idx]
	mov [esi+veh.engineidx],ax
	mov ax,[esi+veh.nextunitidx]

	mov esi,[%$previous]
	mov [esi+veh.nextunitidx],ax

.keepnextunit:
	pop esi
	pop eax
	_ret
endproc

// insert vehicle esi into consist after edi
// in:	 ax=new vehicle idx
//	esi=new vehicle
//	edi=vehicle to be inserted after
// safe:ax
insertveh:
	xchg ax,[edi+veh.nextunitidx]
	mov [esi+veh.nextunitidx],ax

	mov ax,[edi+veh.engineidx]
	mov [esi+veh.engineidx],ax
	ret


// activate random trigger event for vehicle,
// and update random byte if so desired
//
// in:	 al=trigger bit(s)
//	esi=vehicle
// out:	-
// destroys eax
randomtrigger:
	or [esi+veh.newrandom],al

	movzx eax,byte [esi+veh.vehtype]
	mov al,[randomtriggers+eax]

	mov ah,al
	and al,[esi+veh.newrandom]
	jz .nottriggeredyet	// no matching random triggers

	test ah,ah
	jns .anytrigger

	cmp ah,al
	jne .nottriggeredyet

.anytrigger:
	call dword [randomfn]
	mov [esi+veh.random],al
	mov byte [esi+veh.newrandom],0

.nottriggeredyet:
	ret

// end of procs that serve to maintain the veh.engineidx variable


// Get the first day of a given year (with 32-bit date range support)
// in:	EAX=year-1920
// out:	EBX=date
//	CF set if overflow
// destroys EAX,ECX,EDX
yeartodate:
	xor ecx,ecx
	mov cl,100
	xor edx,edx
	div ecx				// EAX = year/100; EDX = year%100

	shld ecx,eax,30			// ECX = year/400 (note: ECX was 100, i.e. the low 2 bits were zero)
	imul ebx,ecx,146097		// accumulate days in EBX
	jc short .quit

	and eax,byte 3
	imul ecx,eax,36525
	cmp dl,80			// 2000, 2100, 2200 etc.
	ja short .leapc_adjust
	dec eax
	js short .leapc_adjusted
.leapc_adjust:
	sub ecx,eax
.leapc_adjusted:
	add ebx,ecx
	jc short .quit

	mov eax,edx
	shr edx,2
	imul edx,1461
	and al,3
	imul ecx,eax,366
	add edx,ecx
	dec eax
	js short .leapadjusted
	sub edx,eax
.leapadjusted:
	add ebx,edx

.quit:
	ret


//
// The following procs deal with saving and restoring some of TTD's internal
// data, including vehicle info.  It is saved once at the initialization of
// the patch, and restored before a new game is started, the scenario editor
// is opened or a game is loaded.
//
// Then, after loading the game, the new vehicle data from .grf files is
// applied, and the postinfoapply proc is called to let the patches do their
// work.
//


struc ttdvehinfo
	.vehtypedata:	resb totalvehtypes*vehtypeinfo_size		// vehtypeinfo structs
	.traindata:	resb NTRAINTYPES*spectraindata_totalsize	// train data
	.rvdata:	resb NROADVEHTYPES*specrvdata_totalsize		// road veh data
	.shipdata:	resb NSHIPTYPES*specshipdata_totalsize		// ship data
	.aircraftdata:	resb NAIRCRAFTTYPES*specplanedata_totalsize	// aircraft data
	.vehtypenames:	resb 256*4					// default vehtype name ptrs
endstruc_32

uvarb ttdvehinfobackup, ttdvehinfo_size

// As a special case, if unifiedmaglev is on, railway vehicle types may have to be converted
// depending on unimaglevmode and the electrifiedrail flag.  Unlike all the other actions,
// this conversion must not be applied more than once.
// To keep the vehtype data consistent also when saved and loaded via the extra type=0 chunk
// (see loadsave.asm) we keep a secondary backup of the data just before the conversion.

uvard vehtypedataconvbackupptr	// 0 if the area is not used (see patches.ah/patchunifiedmaglev)


// in:	EBX=class, 0..3 for railway..aircraft
// out:	ESI->property base
//	ECX=AX=total length
getspecificpropertyarea:
	mov esi,[specificpropertybase+ebx*4]
	mov ecx,[specificpropertylist+ebx*4]
	mov al,[ecx]
	mul byte [vehbnum+ebx]
	movzx ecx,ax
	ret


proc copyinfo
	arg direction
	slocal src_dst,dword,2

	_enter

	mov edx,[%$direction]

	mov edi,ttdvehinfobackup
	mov [%$src_dst],edi
	mov [%$src_dst+4],edi

	mov eax,[vehtypedataptr]
	mov [%$src_dst+edx*4],eax

	mov ecx,totalvehtypes*vehtypeinfo_size
	call .docopy

	xor ebx,ebx
.nextvehtype:
	call getspecificpropertyarea
	mov [%$src_dst+edx*4],esi

	call .docopy

	inc ebx
	cmp ebx,4
	jb .nextvehtype

	mov ah,0x80
	call gettexttableptr
	mov [%$src_dst+edx*4],eax
	mov ch,4
	call .docopy
	_ret

.docopy:
	mov esi,[%$src_dst]
	mov edi,[%$src_dst+4]
	rep movsb
	mov [%$src_dst],esi
	mov [%$src_dst+4],edi
	ret
endproc



	// save all TTD data that is modified in any of preinfoapply,
	// grfinfoapply or postinfoapply
infosave:
	pusha

	param_call copyinfo, 0

	// save default sprite ids in a convenient table
	xor ebx,ebx
	mov edi,defvehsprites

.nextvehtype:
	mov esi,[specificpropertybase+ebx*4]
	movsx eax,byte [specificpropertyofs+ebx]
	movzx ecx,byte [vehbnum+ebx]
	imul eax,ecx
	sub esi,eax
	rep movsb
	inc ebx
	cmp ebx,4
	jb .nextvehtype

	popa
	ret


	// reset the vehicle data
	// called before loading a new game
inforeset:
	pusha
	param_call copyinfo, 1
	call initttdpatchdata
	popa
	ret


	// apply the new info from .grfs
	// called after loading a new game
infoapply:
	pusha
	call preinfoapply
	call setactivegrfs
	mov dword [numactsprites],baseoursprites
	param_call procallsprites,addr(pseudospriteaction_activate)
	call postinfoapply
	popa
	ret


	// reset each of the new TTDPatch vehicle properties
initttdpatchdata:
	mov edi,newvehdata
	mov ecx,newvehdatastruc_size/4
	xor eax,eax
	rep stosd

	// initialize newvehdata.loadamount
	mov edi,loadamount
	mov al,5
	mov cl,NTRAINTYPES+NROADVEHTYPES
	rep stosb
	mov al,10
	mov cl,NSHIPTYPES
	rep stosb
	mov al,20
	mov cl,NAIRCRAFTTYPES
	rep stosb

	mov al,0

	// clear newgrf data
	mov edi,vehids
	mov ecx,256+256/4
	rep stosd

	// train smoke type and vehsorttable
	mov edi,trainsmoketype
	mov ebx,defvehsprites
	mov esi,vehsorttable
	mov cl,NTRAINTYPES
.setnexttrain:
	mov [esi],al
	xlatb
	stosb
	lodsb		// restore unXLATed al and increase ESI
	inc eax
	loop .setnexttrain

	mov cl,256-NTRAINTYPES
	mov edi,esi
.setnextother:
	stosb
	inc eax
	loop .setnextother

	// initialize road vehicle power and weight
	mov esi,rvpowerinit
	mov edi,rvpowers
.initnextset:
	mov cl,7
	rep movsb	// first the busses
	mov cl,27
	lodsd		// then the sets of 3 trucks each
.initnexttriple:
	stosd		// write 4 bytes
	dec edi		// then set edi back so only 3 bytes are stored
	loop .initnexttriple
	test eax,eax
	js .initnextset

	// initialize the sprite translation table
	mov edi,newttdsprites
	xor eax,eax
	mov ecx,totalsprites
.nextsprite:
	stosw
	inc eax
	loop .nextsprite

	// set TTD's default ship refit cargos
	lea edi,[newrefit+(SHIPBASE-ROADVEHBASE)*4]
	xor ecx,ecx
	mov cl,NSHIPTYPES
	mov eax,oldshiprefitlist
	rep stosd

	testflags generalfixes
	jnc .dontswap

	// swap monorail and maglev tunnels; #2435 and #2436
	rol dword [newttdsprites+2435*2],16

.dontswap:
	and word [lasttreepos],0

	// also clear all TTDPatch overridden graphics
	// (i.e. those that have the immutable flag set)
	// that way TTD will use its own graphics again
	// in the scenario/savegame currently being loaded

	mov ebx,[newspritedata]
	mov ebp,[newspritenum]
	imul edx,ebp,19
	add edx,ebx

	xor eax,eax
	xor edi,edi

.checknextsprite:
	cmp [edx+edi],al		// check immutable flag (al is zero)
	je .ttdsprite

	mov [edx+edi],al		// remove immutable flag
	mov dword [ebx+edi*4],eax	// remove from "cache" (it wasn't actually in the cache)

	// and restore sprite info
	pusha
	call reloadspriteheaders
	popa

.ttdsprite:
	inc edi
	cmp edi,totalsprites
	jb .checknextsprite

	// mark the rest of the sprites as present and immutable
	// all the way up to number 16384, so that if TTD "accidentally"
	// accesses such a sprite, it will only show the wrong one and
	// not try to load it from trg1.grf

#if WINTTDX
	mov eax,[spritecacheptr]
	test eax,eax
	jle .done

	mov eax,[eax]
#else
	mov eax,[spritecache]
#endif

	test eax,eax
	jle .done

	add eax,5	// use first entry from sprite cache instead, whatever that is

.setnextsprite:
	cmp edi,ebp
	jb .dosetsprite

.done:
	ret

.dosetsprite:
	mov [ebx+edi*4],eax	// set sprite data cache pointer
	mov byte [edx+edi],1	// set immutable flag

	inc edi
	jmp .setnextsprite


	// things to do before the .grfs modify the vehicles
preinfoapply:
	mov edx,patchflags1
	testflagbase edx

	call initisengine

	// *********

	// reset the railway vehicle sort table so that waggons will end up at the end of the list, after engines
	// (used in the New Vehicles windows)
	testmultiflags unifiedmaglev,newtrains
	jz .notrainsorting

.trainsorting:
	call initrailvehsorttable

	// *********

.notrainsorting:
	testflagbase none
	ret


	// things to do after the .grfs modify the vehicles
postinfoapply:
	testmultiflags morecurrencies
	jz .noeuro

		// add Euro glyph unless it's there already
		// or if the Euro is not to be introduced
	test byte [morecurropts],1<<CURR_OPT_NOEURO
	jnz .noeuro

	mov eax,[newspritedata]
	mov ebx,[newspritenum]

	cmp byte [eax+ebx*4+0x80*2],10
	jnb .noeuro

	mov esi,euroglyph
	call overrideembeddedsprite
	call overrideembeddedsprite
	call overrideembeddedsprite

.noeuro:
	// .grfs or the above may have modified the font, calculate the new width
	push es
	call [setcharwidthtablefn]
	pop es

	// set euro currency symbol to EUR unless Euro glyph is available
	cmp byte [charwidthtables+0x7e],1
	ja .haveeurogrf

	mov dword [currsymsbefore+CURR_EURO*4],"EUR "
	mov dword [currsymsafter+CURR_EURO*4]," EUR"

.haveeurogrf:
	mov edx,patchflags1
	testflagbase edx

	// *********

	testmultiflags newbridgespeeds
	jz .nonewbridgespeeds

.donewbridgespeeds:
	call setbridgespeedlimits

	// *********

.nonewbridgespeeds:
	call initisengine
	call shipsrefittable

	// *********

	testmultiflags electrifiedrail
	jz near .typeconversion

	movzx ebx,byte [unimaglevmode]

	// change build menu sprites and cursors

	mov edi,newttdsprites+1255*2
	mov ecx,4

	movzx eax,word [catenaryspritebase]
	test ax,ax
	jg .haveelrailgrf

	mov eax,0x800004e3	// without graphics, set el.rails to normal railway (1251=0x4e3)
	jmp short .next1	// bit 31 marks it such for later

.haveelrailgrf:
	add eax,elrailsprites.trackicons

	// set monorail build icon to electrified rail
.next1:
	stosw
	inc eax
	loop .next1

	xchg eax,esi

	// set maglev to unimaglev mode
	lea eax,[1251+ebx*4]
	mov cl,4
.next2:
	stosw
	inc eax
	loop .next2

	// somehow last two icons are reverse...
	// need to fix that if unimaglev mode == 1
	cmp ebx,1
	jne .notmrmode

	rol dword [edi-4],16

.notmrmode:
	push edx
	testflagbase none

	lea edx,[eax+8]

	add edi,byte (1267-1263)*2

	// set build cursors
	mov cl,4
	xchg eax,esi

	test eax,eax
	jns .next3

	mov ax,1263
.next3:
	stosw
	inc eax
	loop .next3

	xchg eax,edx
	mov cl,4
.next4:
	stosw
	inc eax
	loop .next4

	// set tunnel icon
	xchg eax,edx

	test eax,eax
	jns .next5

	mov ax,2430

.next5:
	add edi,(2431-1275)*2
	stosw
	inc eax
	mov [edi+4*2],ax

	lea eax,[2430+ebx]
	stosw
	add eax,4
	mov [edi+2*2],ax

	pop edx
	testflagbase edx

	// *********

.typeconversion:
	// Convert between electric, monorail and maglev railway vehicles depending on flags

	// make a backup of the vehtype data if necessary
	mov esi,[vehtypedataptr]
	xor ecx,ecx
	mov edi,[vehtypedataconvbackupptr]
	test edi,edi
	jz .conversionloop

	push esi
	mov cx,totalvehtypes*vehtypeinfo_size/4
	rep movsd
	pop esi

.conversionloop:
	// Note: the same conditions are checked on every loop iteration, over and over again.
	// This is slower, but takes up less space than two separate loops.

	testmultiflags electrifiedrail
	jnz .electrify

	testmultiflags unifiedmaglev
	jz .notypeconversion

	mov al,[unimaglevmode]
	cmp al,3
	je .notypeconversion

	mov bl,3
	sub bl,al			// AL=target type, BL=the 'other' type

	// without electrification:
	// convert one type of maglev to the other
	cmp [esi+vehtypeinfo.traintype],bl
	jne short .next
	bt [isengine],ecx
	jnc short .next
	mov [esi+vehtypeinfo.traintype],al
	jmp short .next

.electrify:
	// with electrification:
	// convert type=1 engines to type=2, and electric type=0 engines to type=1
	mov al,[esi+vehtypeinfo.traintype]
	test al,al
	jne short .notrailroad

	mov al,[trainsmoketype+ecx]
	cmp al,0x28
	jb short .next		// not electric
	cmp al,0x42
	jae short .next		// waggon

	mov byte [esi+vehtypeinfo.traintype],1
	jmp short .next

.notrailroad:
	cmp al,1
	jne short .next
	bt [isengine],ecx
	jnc short .waggon
	mov byte [esi+vehtypeinfo.traintype],2
	jmp short .next

.waggon:
	mov byte [esi+vehtypeinfo.climates],0	// not available

.next:
	add esi,byte vehtypeinfo_size
	inc ecx
	cmp cl,NTRAINTYPES
	jb short .conversionloop

.notypeconversion:
	testflagbase none

	call updatecurrlist
	ret


// New class 0xF (vehtype management) initialization handler
// does additional things before calling the original function
// safe:everything
newvehtypeinit:
	or ax,ax		// the original handler does the same
	jnz .start
	ret

.start:
	push es
	push ds
	pop es
	call inforeset		// reset all vehtype data
	mov byte [activatedefault],1	// new game -> activate all graphics
	call infoapply		// and apply newgrf and other modifications

	pop es

	mov al,1
	mov ebp,dword -1
ovar .oldfn,-4
	call ebp

	cmp dword [spriteerror],0
	je .done

	call grferror

.done:
	ret


//
// --- End of TTD data management procs ---
//


// Critical error handler
// in: EDX->string to display (should be in the OS's native code page, must not contain '$')
criticalerror:
#if WINTTDX
	push byte 0x10
	push dword TTDPatch_msg_title
	push edx
	push byte 0
	call [0x4234a0]		// MessageBoxA

.noerror:
	test byte [didinitialize],2
	jnz .running

	// TTD has not created its window yet
	push 3
	call [0x42344c]		// exitprocess(3)

.running:
	// TTD is already running, perform a standard quit
	mov esi,0x20000
	mov dl,2
	mov bl,1
	mov ebp,[mainhandlertable]
	mov ebp,[ebp+7*8]
	call [ebp+0x10]
	ud2		// we should never get here

var TTDPatch_msg_title, db "TTDPatch",0

#else
	test byte [didinitialize],2
	jz .nocleanupneeded

	push edx
	call [exitcleanup]
	pop edx

.nocleanupneeded:

	// find the end of the string and place a '$' there
	push edx
.findend:
	cmp byte [edx],0
	je .found
	inc edx
	jmp .findend

.found:
	mov byte [edx],'$'
	pop edx

	// display the message and terminate
	mov ah,9		// '$'-terminated string pointed to by EDX
	int 0x21

	mov ax,0x4c03		// exit code 3 = TTDPatch error
	int 0x21
#endif

// Abort TTD with 'out of memory' message
outofmemoryerror:
	mov edx,TTDPatch_out_of_memory
	jmp criticalerror

// this should be eventually loaded from lang/*.h
var TTDPatch_out_of_memory, db "Out of memory!",13,10,0


// Convert location in AX,CX into tile XY offset in ESI
// (assume AX and CX are within the landscape range, i.e. 0x000..0xFFF)
locationtoxy:
	movzx esi,cx
	and esi,byte ~15
	shl esi,8
	or si,ax
	shr esi,4
	ret


// Get ground altitude, but preserve all registers except the result in (E)DX
// in:	AX,CX = X,Y location
// out:	DL = altitude (=height<<3)
//	DH = 1 if under bridge
getgroundalt:
	push esi
	call [getgroundaltitude]
	pop esi
	ret


// Add expenses to a specific company's data
// in:	EBX = amount to subtract from the player's cash
//	AL = player
//	type of expenses must be set in currentexpensetype
// out:	AL = current player
addexpensestoplayer:
	xchg al,[curplayer]
	push edx
	call [addexpenses]
	pop edx
	mov [curplayer],al
	ret
