/* gcc filename -Wall -std=gnu99 -lnetpbm -I/usr/include/netpbm */
/* gcc filename -Wall -std=gnu99 -lnetpbm -I/usr/include/netpbm -S -fverbose-asm */

/* Notes about the original template:
 * - get_timer() uses float division
 * - image_rowbytes(), image_pixel() deserves to be marked inline
 * - code depends on CHAR_BIT==8
 */

#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <pbm.h>

struct image {
	int cols, rows;
	unsigned char *bitmap;
};
static int image_rowbytes(struct image *img);
static int image_pixel(struct image *img, int x, int y);

static struct image *image_load(char *filename);
static void image_save(char *filename, struct image *img);
static void image_free(struct image *img);

typedef int_fast64_t timestamp_t;
static timestamp_t get_timer(void);


int
main(int argc, char *argv[])
{
	if (argc != 3) {
		fprintf(stderr, "%s SRCIMAGE.pbm DSTIMAGE.pbm\n", argv[0]);
		return EXIT_FAILURE;
	}

	struct image *img = image_load(argv[1]);
	timestamp_t t0 = get_timer();

	/*** Your code goes here. ***/

	/* Example: */
	for (int y = 0; y < img->rows; y++) {
		for (int x = 0; x < img->cols; x++)
			putchar(".#"[image_pixel(img, x, y)]);
		putchar('\n');
	}

	/*** Your code ends here. */

	t0 = get_timer() - t0;
	fprintf(stderr, "time spent: %.3f\n", (double) t0/1e6);
	image_save(argv[2], img);
	image_free(img);
	return EXIT_SUCCESS;
}


#if CHAR_BIT != 8
#error char size larger than 8 is not supported
#endif

static inline int
image_rowbytes(struct image *img)
{
	return (img->cols + 7) / 8;
}

static inline int
image_pixel(struct image *img, int x, int y)
{
	return (img->bitmap[image_rowbytes(img) * y + x / 8] >> (7 - x % 8)) & 1;
}

static struct image *
image_load(char *filename)
{
	FILE *f = fopen(filename, "rb");
	if (!f) { perror("load"); exit(EXIT_FAILURE); }

	struct image img; int fmt;
	pbm_readpbminit(f, &img.cols, &img.rows, &fmt);

	img.bitmap = malloc(img.rows * image_rowbytes(&img));
	for (int i = 0; i < img.rows; i++)
		pbm_readpbmrow_packed(f, img.bitmap + i * image_rowbytes(&img),
				img.cols, fmt);

	fclose(f);

	struct image *imga = malloc(sizeof(*imga));
	*imga = img;
	return imga;
}

static void
image_save(char *filename, struct image *img)
{
	FILE *f = fopen(filename, "wb");
	if (!f) { perror("save"); exit(EXIT_FAILURE); }

	pbm_writepbminit(f, img->cols, img->rows, 0);
	for (int i = 0; i < img->rows; i++)
		pbm_writepbmrow_packed(f, img->bitmap + i * image_rowbytes(img),
				img->cols, 0);

	fclose(f);
}

static void
image_free(struct image *img)
{
	free(img->bitmap);
	free(img);
}


static timestamp_t
get_timer(void)
{
	struct timeval t;
	gettimeofday(&t, NULL);
	return 1000000*t.tv_sec + t.tv_usec;
}
